import type { Campaign, CampaignFormState, SequenceStep } from './types';

/**
 * "Translates" a Campaign object from the database into the shape
 * required by our frontend form state.
 */
export function mapCampaignToFormState(campaign: Campaign): CampaignFormState {
  return {
    name: campaign.name ?? '',
    objective: campaign.objective ?? '',
    selectedAudience: campaign.audience ?? null,
    // This is the key transformation: we extract the array
    sequence: campaign.sequence?.steps ?? [],
    senderIds: campaign.senderIds ?? [],
    trackingDomainId: campaign.trackingDomainId,
    warmup: campaign.sender?.warmup ?? false,
    throttling: campaign.sender?.throttling ?? false,
  };
}

/**
 * "Translates" the form state from the frontend back into the data
 * structure required by our backend server action.
 */
export function mapFormStateToCampaignData(formState: CampaignFormState, status: Campaign['status']): Omit<Campaign, 'id' | 'analytics' | 'user_id'> {
  const { name, objective, selectedAudience, sequence, senderIds, trackingDomainId, warmup, throttling } = formState;

  if (!selectedAudience) {
      throw new Error("Audience is not selected");
  }

  return {
    name,
    objective,
    status,
    audienceListId: selectedAudience.id,
    // This is the key transformation: we reconstruct the object
    sequence: { steps: sequence },
    senderIds,
    trackingDomainId,
    sender: { warmup, throttling },
    createdAt: new Date().toISOString(),
    lastModified: new Date().toISOString(),
  };
}
