
'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useCallback } from 'react';
import type { User } from '@/lib/types';
import { getUsers, updateUser as serverUpdateUser, deleteUser as serverDeleteUser } from '@/app/(app)/admin/actions';

interface UserContextType {
  users: User[];
  addUser: (user: User) => void; // Note: addUser might be deprecated if signup action handles it
  updateUser: (id: string, updatedUser: Partial<User>) => Promise<void>;
  deleteUser: (id: string) => Promise<void>;
  fetchUsers: () => Promise<void>;
}

const UserContext = createContext<UserContextType | undefined>(undefined);

export const UserProvider = ({ children }: { children: ReactNode }) => {
  const [users, setUsers] = useState<User[]>([]);

  const fetchUsers = useCallback(async () => {
    try {
        const fetchedUsers = await getUsers();
        setUsers(fetchedUsers);
    } catch (error) {
        console.error("Failed to fetch users:", error);
    }
  }, []);

  useEffect(() => {
    fetchUsers();
  }, [fetchUsers]);


  const addUser = (user: User) => {
    // This is now primarily handled by the server-side signup action.
    // We can keep this for any potential client-side additions or optimistic UI updates.
    setUsers(prev => [...prev, user]);
  };

  const updateUser = async (id: string, updatedUserData: Partial<User>) => {
    try {
      await serverUpdateUser(id, updatedUserData);
      await fetchUsers(); // Re-fetch to ensure sync with DB
    } catch (error) {
      console.error("Failed to update user:", error);
    }
  };

  const deleteUser = async (id: string) => {
    try {
      await serverDeleteUser(id);
      await fetchUsers(); // Re-fetch to ensure sync with DB
    } catch (error) {
      console.error("Failed to delete user:", error);
    }
  };

  return (
    <UserContext.Provider value={{ users, addUser, updateUser, deleteUser, fetchUsers }}>
      {children}
    </UserContext.Provider>
  );
};

export const useUsers = (): UserContextType => {
  const context = useContext(UserContext);
  if (context === undefined) {
    throw new Error('useUsers must be used within a UserProvider');
  }
  return context;
};
