'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useCallback } from 'react';
import type { Campaign } from '@/lib/types';
// --- THIS IS THE KEY: We only import the server actions here ---
import { getCampaigns, createCampaign as createCampaignAction } from '@/app/(app)/campaigns/actions';

interface CampaignContextType {
  campaigns: Campaign[];
  isLoading: boolean;
  addCampaign: (campaignData: Omit<Campaign, 'id' | 'analytics' | 'user_id' | 'audience'>) => Promise<{ success: boolean; message: string }>;
  fetchCampaigns: () => Promise<void>; // Expose fetch for other components to use
  // ... (delete and update would go here too)
}

const CampaignContext = createContext<CampaignContextType | undefined>(undefined);

export const CampaignProvider = ({ children }: { children: ReactNode }) => {
  const [campaigns, setCampaigns] = useState<Campaign[]>([]);
  const [isLoading, setIsLoading] = useState(true);

  // --- THIS FUNCTION FETCHES DATA FROM THE DATABASE ---
  const fetchCampaigns = useCallback(async () => {
    setIsLoading(true);
    try {
      const serverCampaigns = await getCampaigns();
      setCampaigns(serverCampaigns);
    } catch (error) {
      console.error("Failed to fetch campaigns:", error);
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchCampaigns();
  }, [fetchCampaigns]);

  // --- THIS FUNCTION SAVES DATA AND THEN RE-FETCHES ---
  const addCampaign = async (campaignData: Omit<Campaign, 'id' | 'analytics' | 'user_id' | 'audience'>) => {
    try {
      const result = await createCampaignAction(campaignData);
      if (result.success) {
        // After a successful save, immediately fetch the new, complete list
        await fetchCampaigns();
      }
      return result;
    } catch (error) {
      console.error("Failed to add campaign:", error);
      return { success: false, message: "An unexpected error occurred." };
    }
  };

  return (
    <CampaignContext.Provider value={{ campaigns, isLoading, addCampaign, fetchCampaigns }}>
      {children}
    </CampaignContext.Provider>
  );
};

export const useCampaigns = (): CampaignContextType => {
  const context = useContext(CampaignContext);
  if (context === undefined) {
    throw new Error('useCampaigns must be used within a CampaignProvider');
  }
  return context;
};

