'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useCallback } from 'react';
import type { ActivityLog } from '@/lib/types';
import { getActivityLogsAction, addActivityLogAction } from '@/app/(app)/admin/activity-logs/actions';

interface ActivityLogContextType {
  logs: ActivityLog[];
  addLog: (logData: Omit<ActivityLog, 'id' | 'timestamp'>) => Promise<void>;
  fetchLogs: () => Promise<void>;
  isLoading: boolean;
}

const ActivityLogContext = createContext<ActivityLogContextType | undefined>(undefined);

export const ActivityLogProvider = ({ children }: { children: ReactNode }) => {
  const [logs, setLogs] = useState<ActivityLog[]>([]);
  const [isLoading, setIsLoading] = useState(true);

  const fetchLogs = useCallback(async () => {
    setIsLoading(true);
    try {
      const dbLogs = await getActivityLogsAction();
      setLogs(dbLogs);
    } catch (error) {
      console.error('Failed to fetch activity logs:', error);
      setLogs([]);
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchLogs();
  }, [fetchLogs]);

  const addLog = async (logData: Omit<ActivityLog, 'id' | 'timestamp'>) => {
    try {
      const newLog = await addActivityLogAction(logData);
      setLogs(prev => [newLog, ...prev]);
    } catch (error) {
      console.error('Failed to add log to DB:', error);
    }
  };

  return (
    <ActivityLogContext.Provider value={{ logs, addLog, fetchLogs, isLoading }}>
      {children}
    </ActivityLogContext.Provider>
  );
};

export const useActivityLog = (): ActivityLogContextType => {
  const context = useContext(ActivityLogContext);
  if (context === undefined) {
    throw new Error('useActivityLog must be used within an ActivityLogProvider');
  }
  return context;
};
