'use client';

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { PlusCircle, Trash2, FileQuestion, Users, Mails, Send, Flame, Gauge, ChevronDown, Globe, Book } from "lucide-react";
import type { Campaign, CampaignObjective, AudienceList, SequenceStep, Template } from "@/lib/types";
import { useToast } from "@/hooks/use-toast";
import { useAudience } from "@/context/AudienceContext";
import { useTemplates } from "@/context/TemplateContext";
import { Badge } from "@/components/ui/badge";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Separator } from "@/components/ui/separator";
import { getAvailableSenders, createCampaign } from '@/app/(app)/campaigns/actions';
import { getVerifiedTrackingDomains } from '@/app/(app)/settings/domains/actions';
import { SenderSelectionDialog } from '@/app/(app)/campaigns/_components/sender-selection-dialog';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogClose,
} from "@/components/ui/dialog";
import { ScrollArea } from "@/components/ui/scroll-area";
import { TinyMCEEditor } from './tinymce-editor';

// Define the local state for the form
interface CampaignFormState {
  name: string;
  objective: CampaignObjective | '';
  selectedAudience: AudienceList | null;
  sequence: SequenceStep[];
  senderIds: number[];
  trackingDomainId?: number;
  warmup: boolean;
  throttling: boolean;
}

type Sender = { id: number; name: string; service_type: string };
type VerifiedDomain = { id: number; domain_name: string };

function SelectAudienceDropdown({ onSelect, onNew }: { onSelect: (list: AudienceList) => void, onNew: () => void }) {
    const { audienceLists } = useAudience();
    const savedLists = audienceLists.filter(list => list.status === 'Saved');
    return (
      <DropdownMenu>
        <DropdownMenuTrigger asChild><Button variant="outline">Select Audience <ChevronDown className="ml-2 h-4 w-4" /></Button></DropdownMenuTrigger>
        <DropdownMenuContent align="start" className="w-80">
          {savedLists.map(list => ( <DropdownMenuItem key={list.id} onClick={() => onSelect(list)}><div><span>{list.name}</span><span className="text-xs text-muted-foreground">~{list.count.toLocaleString()} recipients</span></div></DropdownMenuItem> ))}
          <DropdownMenuSeparator />
          <DropdownMenuItem onClick={onNew}><PlusCircle className="mr-2 h-4 w-4" />Create New Audience</DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
    );
}

function SelectTemplateDialog({ onSelectTemplate, open, onOpenChange }: { onSelectTemplate: (template: Template) => void, open: boolean, onOpenChange: (open: boolean) => void }) {
    const { templates } = useTemplates();
    const handleSelect = (template: Template) => { onSelectTemplate(template); onOpenChange(false); };
    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl"><DialogHeader><DialogTitle>Select an Email Template</DialogTitle><DialogDescription>Choose a starting point for your email.</DialogDescription></DialogHeader>
                <div className="py-4 max-h-[60vh] overflow-y-auto"><div className="space-y-2">
                    {templates.map(template => ( <button key={template.id} onClick={() => handleSelect(template)} className="w-full text-left p-3 border rounded-lg hover:bg-muted/50"><p className="font-semibold">{template.name}</p><p className="text-sm text-muted-foreground truncate">Subject: {template.subject}</p></button>))}
                    {templates.length === 0 && ( <p className="text-center text-muted-foreground py-8">No templates found.</p> )}
                </div></div>
            </DialogContent>
        </Dialog>
    );
}

export function CreateCampaignDialog({ open, onOpenChange }: { open: boolean, onOpenChange: (open: boolean) => void }) {
  const { toast } = useToast();
  const router = useRouter();

  const [formState, setFormState] = useState<CampaignFormState>({
      name: '', objective: '', selectedAudience: null,
      sequence: [{ id: `step${Date.now()}`, name: 'Initial Email', description: 'The first touchpoint.', delay: 0, condition: 'not-opened', templateId: null, subject: '', body: '' }],
      senderIds: [], trackingDomainId: undefined, warmup: false, throttling: false,
  });

  const { name, objective, warmup, throttling, selectedAudience, sequence, senderIds, trackingDomainId } = formState;

  const [availableSenders, setAvailableSenders] = useState<Sender[]>([]);
  const [verifiedDomains, setVerifiedDomains] = useState<VerifiedDomain[]>([]);
  const [isSenderDialogOpen, setIsSenderDialogOpen] = useState(false);
  const [templateDialogState, setTemplateDialogState] = useState<{ open: boolean; stepId: string | null }>({ open: false, stepId: null });

  useEffect(() => {
    getAvailableSenders().then(setAvailableSenders);
    getVerifiedTrackingDomains().then(setVerifiedDomains);
  }, []);

  const setFormField = <K extends keyof CampaignFormState>(field: K, value: CampaignFormState[K]) => setFormState(prev => ({ ...prev, [field]: value }));
  const setSelectedAudience = (list: AudienceList | null) => setFormState(prev => ({ ...prev, selectedAudience: list }));
  const setSequence = (sequence: SequenceStep[]) => setFormState(prev => ({...prev, sequence}));
  const setSenderIds = (ids: number[]) => setFormState(prev => ({...prev, senderIds: ids}));

  const addSequenceStep = () => setSequence([...sequence, { id: `step${Date.now()}`, name: `Follow-up ${sequence.length}`, description: 'A follow-up email.', delay: 3, condition: 'not-opened', templateId: null, subject: '', body: '' }]);
  const removeSequenceStep = (id: string) => {
    if (sequence.length > 1) setSequence(sequence.filter(step => step.id !== id));
    else toast({ variant: 'destructive', title: 'A campaign must have at least one step.' });
  };
  const handleUpdateStep = (id: string, field: keyof SequenceStep, value: any) => setSequence(sequence.map(step => (step.id === id ? { ...step, [field]: value } : step)));

  const handleSelectTemplate = (template: Template) => {
    if (templateDialogState.stepId) {
        const newSequence = sequence.map(step =>
            step.id === templateDialogState.stepId ? { ...step, templateId: template.id, subject: template.subject, body: template.body } : step
        );
        setSequence(newSequence);
    }
    setTemplateDialogState({ open: false, stepId: null });
  };

  const handleCreateCampaign = async (status: 'Draft' | 'Scheduled') => {
    if (!name || !objective || !selectedAudience || senderIds.length === 0 || sequence.some(s => !s.subject || !s.body)) {
      toast({ variant: "destructive", title: "Missing Information", description: "Please complete all fields, including email content for each step." });
      return;
    }

    const newCampaignData: Omit<Campaign, 'id' | 'analytics' | 'user_id'> = {
      name, objective, status,
      audienceListId: selectedAudience!.id,
      sequence: { steps: sequence },
      senderIds, trackingDomainId,
      sender: { warmup, throttling },
      createdAt: new Date().toISOString(),
      lastModified: new Date().toISOString(),
    };
    const result = await createCampaign(newCampaignData);
    if (result.success) {
      toast({ title: "Campaign Created!", description: result.message });
      onOpenChange(false);
    } else {
      toast({ variant: "destructive", title: "Save Failed", description: result.message });
    }
  };

  const selectedSenderObjects = availableSenders.filter(s => senderIds.includes(s.id));

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl h-[90vh] flex flex-col">
        <DialogHeader><DialogTitle>Create New Campaign</DialogTitle><DialogDescription>Follow the steps to launch your next successful campaign.</DialogDescription></DialogHeader>
        <ScrollArea className="flex-grow pr-6 -mr-6">
            <div className="space-y-8 p-1">
                <Card>
                    <CardHeader>
                        {/* --- ENHANCED HEADER --- */}
                        <div className="flex items-center gap-4">
                            <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><FileQuestion className="h-6 w-6" /></div>
                            <div><CardTitle>Step 1: Objective</CardTitle><CardDescription>Define the goal of your campaign.</CardDescription></div>
                        </div>
                    </CardHeader>
                    <CardContent className="grid md:grid-cols-2 gap-6">
                        <div className="grid gap-2"><Label htmlFor="campaign-name-dialog">Campaign Name</Label><Input id="campaign-name-dialog" value={name} onChange={(e) => setFormField('name', e.target.value)} /></div>
                        <div className="grid gap-2"><Label htmlFor="campaign-objective-dialog">Campaign Objective</Label><Select value={objective} onValueChange={(value) => setFormField('objective', value as CampaignObjective)}><SelectTrigger id="campaign-objective-dialog"><SelectValue placeholder="Select an objective" /></SelectTrigger><SelectContent><SelectItem value="Lead Nurturing">Lead Nurturing</SelectItem><SelectItem value="Sales Conversion">Sales Conversion</SelectItem><SelectItem value="Engagement/Retention">Engagement/Retention</SelectItem><SelectItem value="Brand Awareness">Brand Awareness</SelectItem></SelectContent></Select></div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader>
                        <div className="flex items-center gap-4">
                            <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Users className="h-6 w-6" /></div>
                            <div><CardTitle>Step 2: Audience</CardTitle><CardDescription>Select who will receive this campaign.</CardDescription></div>
                        </div>
                    </CardHeader>
                    <CardContent>
                        {selectedAudience ? (<Card className="bg-muted/30"><CardHeader><div className="flex justify-between items-center"><div><CardTitle className="text-base">{selectedAudience.name}</CardTitle><CardDescription>~{selectedAudience.count.toLocaleString()} recipients</CardDescription></div><SelectAudienceDropdown onSelect={setSelectedAudience} onNew={() => router.push('/audience/new?redirect=/campaigns')}/></div></CardHeader></Card>) : (<div className="flex flex-col items-center justify-center text-center gap-4 p-8 border-2 border-dashed rounded-lg"><p className="text-muted-foreground">No audience list selected.</p><SelectAudienceDropdown onSelect={setSelectedAudience} onNew={() => router.push('/audience/new?redirect=/campaigns')}/></div>)}
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader>
                        <div className="flex items-center gap-4">
                            <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Mails className="h-6 w-6" /></div>
                            <div><CardTitle>Step 3: Email Sequence</CardTitle><CardDescription>Design the flow and content of your campaign.</CardDescription></div>
                        </div>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        {sequence.map((step, index) => (
                            <div key={step.id} className="p-4 border rounded-lg space-y-4">
                                <div className="flex justify-between items-start">
                                    <div className="grid gap-1.5 flex-1"><Label htmlFor={`step-name-${step.id}`}>Step Name</Label><Input id={`step-name-${step.id}`} value={step.name} onChange={(e) => handleUpdateStep(step.id, 'name', e.target.value)} className="text-base font-semibold"/></div>
                                    <Button variant="ghost" size="icon" onClick={() => removeSequenceStep(step.id)} className="ml-4"><Trash2 className="h-4 w-4" /></Button>
                                </div>
                                <Separator/>
                                {/* --- THIS IS THE KEY VISUAL UPGRADE --- */}
                                <Button variant="outline" size="sm" onClick={() => setTemplateDialogState({ open: true, stepId: step.id })}>
                                    <Book className="mr-2 h-4 w-4" /> Load from Template
                                </Button>
                                <div className="space-y-2"><Label htmlFor={`step-subject-${step.id}`}>Subject Line</Label><Input id={`step-subject-${step.id}`} value={step.subject} onChange={(e) => handleUpdateStep(step.id, 'subject', e.target.value)} placeholder="Enter email subject..."/></div>
                                <div className="space-y-2"><Label>Email Body</Label><TinyMCEEditor value={step.body} onEditorChange={(newContent) => handleUpdateStep(step.id, 'body', newContent)}/></div>
                                {index > 0 && (<div className="grid gap-2 pt-4 border-t"><Label>Logic</Label><div className="flex items-center gap-2 flex-wrap"><span>Send</span><Input type="number" value={step.delay} onChange={(e) => handleUpdateStep(step.id, 'delay', parseInt(e.target.value))} className="w-16 h-9"/><span>days after previous step, if recipient</span><Select value={step.condition} onValueChange={(v) => handleUpdateStep(step.id, 'condition', v)}><SelectTrigger className="w-[150px] h-9"><SelectValue/></SelectTrigger><SelectContent><SelectItem value="not-opened">has not opened</SelectItem><SelectItem value="not-clicked">has not clicked</SelectItem><SelectItem value="opened-no-click">opened but not clicked</SelectItem></SelectContent></Select></div></div>)}
                            </div>
                        ))}
                        <Button variant="outline" className="w-full" onClick={addSequenceStep}><PlusCircle className="mr-2 h-4 w-4"/>Add Step</Button>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <div className="flex items-center gap-4">
                            <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Send className="h-6 w-6" /></div>
                            <div><CardTitle>Step 4: Sender & Delivery</CardTitle><CardDescription>Configure how emails will be sent and tracked.</CardDescription></div>
                        </div>
                    </CardHeader>
                    <CardContent className="grid md:grid-cols-2 gap-8"><div className="grid gap-4"><Label className="font-semibold">Sending Infrastructure</Label><p className="text-sm text-muted-foreground">Select one or more verified senders for rotation.</p><Button type="button" variant="outline" onClick={() => setIsSenderDialogOpen(true)} className="max-w-xs"><PlusCircle className="mr-2 h-4 w-4" />Select Senders ({senderIds.length} selected)</Button>{selectedSenderObjects.length > 0 && (<div className="p-3 border rounded-md space-y-2 bg-muted/50"><h4 className="text-xs font-medium text-muted-foreground">Selected:</h4><div className="flex flex-wrap gap-1.5">{selectedSenderObjects.map(sender => (<Badge key={sender.id} variant="secondary">{sender.name}</Badge>))}</div></div>)}</div><div className="grid gap-6"><div className="grid gap-2"><Label className="font-semibold flex items-center gap-2"><Globe className="h-4 w-4" /> Tracking Domain</Label><p className="text-sm text-muted-foreground">Choose a verified domain for tracking.</p><Select value={trackingDomainId?.toString()} onValueChange={(value) => setFormField('trackingDomainId', Number(value))} disabled={verifiedDomains.length === 0}><SelectTrigger><SelectValue placeholder="Select a tracking domain..." /></SelectTrigger><SelectContent>{verifiedDomains.map(domain => (<SelectItem key={domain.id} value={domain.id.toString()}>{domain.domain_name}</SelectItem>))}</SelectContent></Select>{verifiedDomains.length === 0 && <p className="text-xs text-red-500">No verified tracking domains found.</p>}</div><div className="flex items-start space-x-3 space-y-0 rounded-md border p-4"><Flame className="h-5 w-5 mt-1 text-muted-foreground"/><div className="space-y-1 leading-none"><div className="flex items-center gap-2"><Checkbox id="warm-up-dialog" checked={warmup} onCheckedChange={(checked) => setFormField('warmup', !!checked)} /><Label htmlFor="warm-up-dialog">Enable Warm-up</Label></div></div></div><div className="flex items-start space-x-3 space-y-0 rounded-md border p-4"><Gauge className="h-5 w-5 mt-1 text-muted-foreground"/><div className="space-y-1 leading-none"><div className="flex items-center gap-2"><Checkbox id="throttling-dialog" checked={throttling} onCheckedChange={(checked) => setFormField('throttling', !!checked)} /><Label htmlFor="throttling-dialog">Enable Throttling</Label></div></div></div></div></CardContent>
                </Card>
            </div>
        </ScrollArea>
        <DialogFooter className="pt-4 border-t flex-shrink-0">
            <DialogClose asChild><Button variant="ghost">Cancel</Button></DialogClose>
            <Button variant="outline" onClick={() => handleCreateCampaign('Draft')}>Save as Draft</Button>
            <Button onClick={() => handleCreateCampaign('Scheduled')}>Launch Campaign</Button>
        </DialogFooter>

      <SenderSelectionDialog open={isSenderDialogOpen} onOpenChange={setIsSenderDialogOpen} availableSenders={availableSenders} selectedSenderIds={senderIds} onSave={setSenderIds} />
      <SelectTemplateDialog open={templateDialogState.open} onOpenChange={(isOpen) => setTemplateDialogState({ open: isOpen, stepId: null })} onSelectTemplate={handleSelectTemplate} />
    </DialogContent>
    </Dialog>
  );
}
