'use client';

import { useState, Suspense } from 'react';
import Link from 'next/link';
import { useSearchParams, useRouter } from 'next/navigation';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Logo } from '@/components/logo';
import { useToast } from '@/hooks/use-toast';
import { resetPassword } from '../forgot-password/actions';
import { AnimatedText } from '@/components/auth/animated-text';


function ResetPasswordForm() {
    const { toast } = useToast();
    const router = useRouter();
    const searchParams = useSearchParams();
    const token = searchParams ? searchParams.get('token') : null;
    
    const [newPassword, setNewPassword] = useState('');
    const [confirmPassword, setConfirmPassword] = useState('');
    const [isSubmitting, setIsSubmitting] = useState(false);
    
    if (!token) {
        return (
          <div className="flex items-center justify-center py-12">
            <div className="mx-auto grid w-[350px] gap-6 border p-6 rounded-lg">
                 <div className="grid gap-2 text-center">
                    <h1 className="text-3xl font-bold">Invalid Token</h1>
                    <p className="text-balance text-muted-foreground">The password reset link is invalid or has expired.</p>
                </div>
                 <Link href="/forgot-password">
                    <Button className="w-full">Request a new link</Button>
                </Link>
            </div>
          </div>
        )
    }
    
    const handleResetPassword = async (e: React.FormEvent) => {
        e.preventDefault();
        if (newPassword !== confirmPassword) {
        toast({
            variant: 'destructive',
            title: 'Passwords do not match',
            description: 'Please ensure both passwords are the same.',
        });
        return;
        }
        setIsSubmitting(true);
        const result = await resetPassword(token, newPassword);
        if (result.success) {
            toast({
                title: 'Password Reset Successful',
                description: 'You can now log in with your new password.',
            });
            router.push('/login');
        } else {
            toast({
                variant: 'destructive',
                title: 'Error',
                description: result.error,
            });
            setIsSubmitting(false);
        }
    };
    
     return (
        <div className="flex items-center justify-center py-12">
            <div className="mx-auto grid w-[350px] gap-6 border p-6 rounded-lg">
                <div className="grid gap-2 text-center">
                    <h1 className="text-3xl font-bold">Reset Your Password</h1>
                    <p className="text-balance text-muted-foreground">
                        Enter your new password below.
                    </p>
                </div>
                    <form onSubmit={handleResetPassword}>
                    <div className="grid gap-4">
                        <div className="grid gap-2">
                            <Label htmlFor="new-password">New Password</Label>
                            <Input
                                id="new-password"
                                type="password"
                                required
                                value={newPassword}
                                onChange={(e) => setNewPassword(e.target.value)}
                                disabled={isSubmitting}
                            />
                        </div>
                        <div className="grid gap-2">
                            <Label htmlFor="confirm-password">Confirm New Password</Label>
                            <Input
                                id="confirm-password"
                                type="password"
                                required
                                value={confirmPassword}
                                onChange={(e) => setConfirmPassword(e.target.value)}
                                disabled={isSubmitting}
                            />
                        </div>
                        <Button type="submit" className="w-full" disabled={isSubmitting}>
                            {isSubmitting ? 'Resetting...' : 'Reset Password'}
                        </Button>
                    </div>
                </form>
            </div>
        </div>
    );
}

export default function ResetPasswordPage() {
    return (
        <div className="w-full lg:grid lg:min-h-screen lg:grid-cols-2">
            <div className="hidden bg-primary lg:flex flex-col items-center justify-between p-8 text-primary-foreground">
                <div className="flex flex-col items-center text-center max-w-md mt-auto">
                    <Logo size="large" />
                    <AnimatedText />
                </div>
                <div className="mt-auto text-xs text-primary-foreground/70">
                  v1.0.0
                </div>
            </div>
            <Suspense fallback={<div>Loading...</div>}>
                <ResetPasswordForm />
            </Suspense>
        </div>
    );
}
