

'use client';

import Link from "next/link"
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button"
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Logo } from "@/components/logo"
import { useToast } from "@/hooks/use-toast";
import { login } from "./actions";
import { UserProvider } from "@/context/UserContext";
import { ActivityLogProvider } from "@/context/ActivityLogContext";
import { AnimatedText } from "@/components/auth/animated-text";

function LoginPageContent() {
  const router = useRouter();
  const { toast } = useToast();
  
  const handleLogin = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const result = await login(formData);

    if (result.success) {
        const user = result.user!;
        if (typeof window !== 'undefined') {
            sessionStorage.setItem('userId', user.id);
            sessionStorage.setItem('userRole', user.role);
            sessionStorage.setItem('userName', user.name);
            sessionStorage.setItem('userEmail', user.email);
            if (user.id === 'ADMIN001') {
                 sessionStorage.setItem('profilePic', user.profile_pic_url ?? 'https://placehold.co/100x100/663399/FFFFFF/png?text=A');
            } else if (user.profile_pic_url) {
                 sessionStorage.setItem('profilePic', user.profile_pic_url);
            }
        }
        window.dispatchEvent(new Event('sessionStorageChanged'));
        const welcomeMessage = user.role === 'Admin' ? 'Welcome back, Admin!' : `Welcome back, ${user.name}!`;
        toast({ title: 'Login Successful', description: welcomeMessage });
        
        if (user.role === 'Admin') {
            router.push('/admin');
        } else {
            router.push('/campaigns');
        }
    } else {
        toast({ variant: 'destructive', title: 'Login Failed', description: result.error });
    }
  };

  return (
    <div className="w-full lg:grid lg:min-h-screen lg:grid-cols-2">
      <div className="hidden bg-primary lg:flex flex-col items-center justify-between p-8 text-primary-foreground">
        <div className="flex flex-col items-center text-center max-w-md mt-auto">
            <Logo size="large" />
            <AnimatedText />
        </div>
        <div className="mt-auto text-xs text-primary-foreground/70">
          v1.0.0
        </div>
      </div>
      <div className="flex items-center justify-center py-12">
        <div className="mx-auto grid w-[350px] gap-6 border p-6 rounded-lg">
            <div className="grid gap-2 text-center">
                <h1 className="text-3xl font-bold">Welcome back</h1>
                <p className="text-balance text-muted-foreground">
                    Enter your credentials to login
                </p>
            </div>
            <form onSubmit={handleLogin}>
                <div className="grid gap-4">
                    <div className="grid gap-2">
                    <Label htmlFor="email">Email</Label>
                    <Input
                        id="email"
                        name="email"
                        type="email"
                        placeholder="m@example.com"
                        required
                    />
                    </div>
                    <div className="grid gap-2">
                    <div className="flex items-center">
                        <Label htmlFor="password">Password</Label>
                        <Link
                        href="/forgot-password"
                        className="ml-auto inline-block text-sm underline"
                        >
                        Forgot your password?
                        </Link>
                    </div>
                    <Input 
                        id="password" 
                        name="password"
                        type="password" 
                        required />
                    </div>
                    <Button type="submit" className="w-full">
                    Login
                    </Button>
                </div>
            </form>
            <div className="mt-4 text-center text-sm">
                Don&apos;t have an account?{" "}
                <Link href="/signup" className="underline">
                    Sign up
                </Link>
            </div>
        </div>
      </div>
    </div>
  )
}

export default function LoginPage() {
    return (
        <UserProvider>
            <ActivityLogProvider>
                <LoginPageContent />
            </ActivityLogProvider>
        </UserProvider>
    )
}
