
'use server';

import { findUserByEmail, updateUser } from '@/lib/db';
import crypto from 'crypto';
import nodemailer from 'nodemailer';
import { findUserByResetToken } from '@/lib/db';
import bcrypt from 'bcryptjs';

const transporter = nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: Number(process.env.SMTP_PORT),
    secure: process.env.SMTP_SECURE === 'true',
    auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS,
    },
});

export async function sendResetLink(email: string): Promise<{ success: boolean; error?: string }> {
  if (!email) {
    return { success: false, error: 'Email is required.' };
  }

  try {
    const user = await findUserByEmail(email);
    if (!user) {
      // Don't reveal that the user does not exist for security reasons.
      // We'll log it on the server and return success to the client.
      console.log(`Password reset requested for non-existent user: ${email}.`);
      return { success: true };
    }

    const token = crypto.randomBytes(32).toString('hex');
    // Hash the token before storing it in the database for added security
    const hashedToken = crypto.createHash('sha256').update(token).digest('hex');

    const expires = new Date();
    expires.setHours(expires.getHours() + 1); // Token expires in 1 hour

    await updateUser(user.id, {
        reset_token: hashedToken,
        reset_token_expires: expires,
    });
    
    const resetUrl = `${process.env.NEXT_PUBLIC_APP_URL}/reset-password?token=${token}`;
    
    const mailOptions = {
        from: '"SmartMail AI" <no-reply@smartmailai.com>', // Use a no-reply address from your domain
        to: user.email,
        subject: 'Reset your password for SmartMail AI',
        html: `<p>You are receiving this because you (or someone else) have requested the reset of the password for your account.</p>
               <p>Please click on the following link, or paste this into your browser to complete the process within one hour of receiving it:</p>
               <p><a href="${resetUrl}">${resetUrl}</a></p>
               <p>If you did not request this, please ignore this email and your password will remain unchanged.</p>`
    };

    await transporter.sendMail(mailOptions);
    
    return { success: true };
  } catch (error) {
      console.error('Error sending password reset link:', error);
      // Provide a generic error message to the user
      return { success: false, error: 'An error occurred while sending the reset link.' };
  }
}

export async function resetPassword(token: string, newPassword: string): Promise<{ success: boolean; error?: string }> {
    if (!token || !newPassword) {
        return { success: false, error: 'Token and new password are required.' };
    }

    if(newPassword.length < 6) {
        return { success: false, error: 'Password must be at least 6 characters long.' };
    }
    
    try {
        const hashedToken = crypto.createHash('sha256').update(token).digest('hex');
        const user = await findUserByResetToken(hashedToken);

        if (!user) {
            return { success: false, error: 'Password reset token is invalid or has expired.' };
        }

        const hashedPassword = await bcrypt.hash(newPassword, 10);
        
        await updateUser(user.id, {
            password: hashedPassword,
            reset_token: null,
            reset_token_expires: null,
        });
        
        return { success: true };

    } catch (error) {
        console.error('Error resetting password:', error);
        return { success: false, error: 'An internal server error occurred.' };
    }
}
