'use client';

import Link from "next/link";
import React, { useState, useMemo, useCallback, useEffect } from "react";
import { PlusCircle, Search, MoreHorizontal, Edit, Trash2, Eye, Download, ArrowDownAZ, ArrowUpAZ, Clock, ChevronDown, Upload, Wand2, Plus, Save, LayoutGrid, List } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { PageHeader } from "@/components/page-header";
import { Input } from "@/components/ui/input";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { useToast } from "@/hooks/use-toast";
import type { Template } from "@/lib/types";
import { useTemplates } from "@/context/TemplateContext";
import { format } from 'date-fns';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import { Separator } from "@/components/ui/separator";
import { TinyMCEEditor } from '@/components/tinymce-editor';




// --- DIALOGS ---
function ManualCreateDialog({ open, onOpenChange, onSave }: { open: boolean, onOpenChange: (open: boolean) => void, onSave: (template: Omit<Template, 'id' | 'userId' | 'created_at' | 'last_modified'>) => void }) {
    const { toast } = useToast();
    const [name, setName] = useState('');
    const [subject, setSubject] = useState('');
    // --- THIS STATE NOW HOLDS HTML ---
    const [body, setBody] = useState('');

    const handleSave = () => {
        if (!name || !subject || !body) {
            toast({ variant: 'destructive', title: 'Missing Fields', description: 'Template Name, Subject, and Body are required.'});
            return;
        }
        // No need to join parts, the body is already complete HTML
        onSave({ name, subject, body });

        // Reset form fields and close the dialog
        setName('');
        setSubject('');
        setBody('');
        onOpenChange(false);
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            {/* --- THIS IS THE FIX --- */}
            {/* ...and the onOpenAutoFocus prop is added here. */}
            <DialogContent className="sm:max-w-[825px]" onOpenAutoFocus={(e) => e.preventDefault()}>
                <DialogHeader>
                    <DialogTitle>Create New Template Manually</DialogTitle>
                    <DialogDescription>
                        Fill in the details to create your template. Click save when you're done.
                    </DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4 max-h-[70vh] overflow-y-auto pr-4">
                    <div className="grid gap-2">
                        <Label htmlFor="manual-template-name">Template Name</Label>
                        <Input id="manual-template-name" value={name} onChange={(e) => setName(e.target.value)} placeholder="e.g., Q1 Follow-Up Email" />
                    </div>
                    <div className="grid gap-2">
                        <Label htmlFor="manual-template-subject">Subject</Label>
                        <Input id="manual-template-subject" value={subject} onChange={(e) => setSubject(e.target.value)} placeholder="Catchy subject line" />
                    </div>

                    {/* --- THIS IS THE KEY UPGRADE --- */}
                    <div className="grid gap-2">
                        <Label>Body</Label>
                        <TinyMCEEditor
                            value={body}
                            onEditorChange={setBody}
                        />
                    </div>
                </div>
                <DialogFooter>
                    <Button variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
                    <Button onClick={handleSave}><Save className="mr-2 h-4 w-4"/>Save Template</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function EditTemplateDialog({ template, open, onOpenChange, onSave }: { template: Template | null, open: boolean, onOpenChange: (open: boolean) => void, onSave: (template: Template) => void }) {
    const [name, setName] = useState('');
    const [subject, setSubject] = useState('');
    const [body, setBody] = useState('');

    useEffect(() => {
        if (template) {
            setName(template.name);
            setSubject(template.subject);
            setBody(template.body);
        }
    }, [template]);

    if (!template) return null;

    const handleSave = () => {
        onSave({ ...template, name, subject, body });
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            {/* --- THIS IS THE FIX --- */}
            {/* ...and the onOpenAutoFocus prop is added here. */}
            <DialogContent className="sm:max-w-[825px]" onOpenAutoFocus={(e) => e.preventDefault()}>
                <DialogHeader><DialogTitle>Edit Template</DialogTitle><DialogDescription>Make changes to your template.</DialogDescription></DialogHeader>
                <div className="grid gap-4 py-4 max-h-[70vh] overflow-y-auto pr-4">
                    <div className="grid gap-2"><Label htmlFor="template-name-edit">Template Name</Label><Input id="template-name-edit" value={name} onChange={(e) => setName(e.target.value)} /></div>
                    <div className="grid gap-2"><Label htmlFor="template-subject-edit">Subject</Label><Input id="template-subject-edit" value={subject} onChange={(e) => setSubject(e.target.value)} /></div>
                    <div className="grid gap-2"><Label>Body</Label><TinyMCEEditor value={body} onEditorChange={setBody} /></div>
                </div>
                <DialogFooter><Button variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button><Button onClick={handleSave}>Save Changes</Button></DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

// --- DISPLAY COMPONENTS ---
function TemplateCard({ template, onActionSelect }: { template: Template, onActionSelect: (template: Template, action: 'delete' | 'edit') => void }) {
    return (
        <Card className="flex flex-col">
            <CardHeader>
                <CardTitle>{template.name}</CardTitle>
                <CardDescription>Subject: {template.subject}</CardDescription>
            </CardHeader>
            <CardContent className="flex-grow">
                 <div className="prose prose-sm dark:prose-invert max-h-24 overflow-hidden relative text-muted-foreground">
                    <div dangerouslySetInnerHTML={{ __html: template.body || '<p>No content.</p>' }} />
                    <div className="absolute bottom-0 left-0 w-full h-12 bg-gradient-to-t from-card to-transparent" />
                </div>
            </CardContent>
            <Separator />
            <CardFooter className="flex justify-between items-center pt-4">
                <p className="text-xs text-muted-foreground">
                    Last modified: {format(new Date(template.last_modified), 'PPP')}
                </p>
                <DropdownMenu>
                    <DropdownMenuTrigger asChild><Button variant="ghost" size="sm"><MoreHorizontal className="h-4 w-4" /></Button></DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                        <DropdownMenuItem onClick={() => onActionSelect(template, 'edit')}><Edit className="mr-2 h-4 w-4" /> Edit</DropdownMenuItem>
                        <DropdownMenuItem className="text-destructive" onClick={() => onActionSelect(template, 'delete')}><Trash2 className="mr-2 h-4 w-4" /> Delete</DropdownMenuItem>
                    </DropdownMenuContent>
                </DropdownMenu>
            </CardFooter>
        </Card>
    );
}

function TemplateList({ templates, onActionSelect }: { templates: Template[], onActionSelect: (template: Template, action: 'delete' | 'edit') => void }) {
    return (
        <Table>
            <TableHeader><TableRow><TableHead>Template Name</TableHead><TableHead>Subject</TableHead><TableHead>Last Modified</TableHead><TableHead className="text-right">Actions</TableHead></TableRow></TableHeader>
            <TableBody>
                {templates.length > 0 ? (
                    templates.map((template) => (
                        <TableRow key={template.id}>
                            <TableCell className="font-medium">{template.name}</TableCell>
                            <TableCell>{template.subject}</TableCell>
                            <TableCell>{format(new Date(template.last_modified), 'PPP')}</TableCell>
                            <TableCell className="text-right">
                                <DropdownMenu>
                                    <DropdownMenuTrigger asChild><Button variant="ghost" className="h-8 w-8 p-0"><MoreHorizontal className="h-4 w-4" /></Button></DropdownMenuTrigger>
                                    <DropdownMenuContent align="end">
                                        <DropdownMenuItem onClick={() => onActionSelect(template, 'edit')}><Edit className="mr-2 h-4 w-4" /> Edit</DropdownMenuItem>
                                        <DropdownMenuItem className="text-destructive" onClick={() => onActionSelect(template, 'delete')}><Trash2 className="mr-2 h-4 w-4" /> Delete</DropdownMenuItem>
                                    </DropdownMenuContent>
                                </DropdownMenu>
                            </TableCell>
                        </TableRow>
                    ))
                ) : (
                    <TableRow><TableCell colSpan={4} className="h-24 text-center">No templates found.</TableCell></TableRow>
                )}
            </TableBody>
        </Table>
    );
}

export default function TemplatesPage() {
    const { templates, deleteTemplate, updateTemplate, addTemplate, isLoading } = useTemplates();
    const [viewMode, setViewMode] = useState<'card' | 'list'>('card'); // <-- RESTORED
    const [searchTerm, setSearchTerm] = useState("");
    const [templateToDelete, setTemplateToDelete] = useState<Template | null>(null);
    const [isAlertOpen, setIsAlertOpen] = useState(false);
    const [templateToEdit, setTemplateToEdit] = useState<Template | null>(null);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isManualCreateOpen, setIsManualCreateOpen] = useState(false);
    const { toast } = useToast();

    useEffect(() => { // <-- RESTORED for view mode
        const savedView = localStorage.getItem('templateViewMode');
        if (savedView === 'card' || savedView === 'list') {
            setViewMode(savedView);
        }
    }, []);

    const handleViewChange = (value: 'card' | 'list') => { // <-- RESTORED for view mode
        if (value) {
            setViewMode(value);
            localStorage.setItem('templateViewMode', value);
        }
    };

    const handleAction = (template: Template, action: 'delete' | 'edit') => {
        if (action === 'delete') {
            setTemplateToDelete(template);
            setIsAlertOpen(true);
        } else if (action === 'edit') {
            setTemplateToEdit(template);
            setIsEditOpen(true);
        }
    };

    const handleDeleteConfirm = async () => {
        if (templateToDelete) {
            const result = await deleteTemplate(templateToDelete.id);
            toast({ title: result.success ? 'Template Deleted' : 'Error', description: result.message, variant: result.success ? 'default' : 'destructive'});
            setIsAlertOpen(false);
            setTemplateToDelete(null);
        }
    };

    const handleUpdateConfirm = async (updatedTemplate: Template) => {
        const result = await updateTemplate(updatedTemplate.id, updatedTemplate);
        toast({ title: result.success ? 'Template Updated' : 'Error', description: result.message, variant: result.success ? 'default' : 'destructive'});
        setIsEditOpen(false);
        setTemplateToEdit(null);
    };

    const handleManualCreateConfirm = async (templateData: Omit<Template, 'id' | 'userId' | 'created_at' | 'last_modified'>) => {
        const result = await addTemplate(templateData);
        if (result.success) {
            toast({ title: 'Template Created', description: `"${result.template.name}" has been saved.` });
        } else {
            toast({ variant: 'destructive', title: 'Error', description: result.message });
        }
    };

    const sortedTemplates = useMemo(() => {
        return templates.filter(t => t.name.toLowerCase().includes(searchTerm.toLowerCase()));
    }, [templates, searchTerm]);

    return (
        <>
            <PageHeader
                title="Templates"
                description="Manage your saved email templates."
            >
                <div className="flex gap-2">
                    {/* --- VIEW TOGGLE RESTORED --- */}
                    <TooltipProvider>
                        <ToggleGroup type="single" value={viewMode} onValueChange={handleViewChange}>
                            <Tooltip>
                                <TooltipTrigger asChild><ToggleGroupItem value="card" aria-label="Card view"><LayoutGrid className="h-4 w-4" /></ToggleGroupItem></TooltipTrigger>
                                <TooltipContent><p>Card View</p></TooltipContent>
                            </Tooltip>
                            <Tooltip>
                                <TooltipTrigger asChild><ToggleGroupItem value="list" aria-label="List view"><List className="h-4 w-4" /></ToggleGroupItem></TooltipTrigger>
                                <TooltipContent><p>List View</p></TooltipContent>
                            </Tooltip>
                        </ToggleGroup>
                    </TooltipProvider>
                    {/* --- DROPDOWN RESTORED --- */}
                    <DropdownMenu>
                        <DropdownMenuTrigger asChild><Button><PlusCircle className="mr-2 h-4 w-4" />Create Template</Button></DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                            <DropdownMenuItem asChild><Link href="/ai-drafts/new"><Wand2 className="mr-2 h-4 w-4" />Create with AI</Link></DropdownMenuItem>
                            <DropdownMenuItem onSelect={() => setIsManualCreateOpen(true)}><Plus className="mr-2 h-4 w-4" />Create Manually</DropdownMenuItem>
                        </DropdownMenuContent>
                    </DropdownMenu>
                    <Button variant="outline" disabled><Upload className="mr-2 h-4 w-4" /> Import</Button>
                </div>
            </PageHeader>
            <Card>
                <CardHeader>
                    <CardTitle>Template List</CardTitle>
                    <div className="relative mt-4">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                        <Input placeholder="Search templates..." className="pl-10" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                    </div>
                </CardHeader>
                <CardContent>
                    {isLoading ? (
                        <div className="text-center p-8 text-muted-foreground">Loading templates...</div>
                    ) : (
                        <>
                            {sortedTemplates.length > 0 ? (
                                viewMode === 'list' ? (
                                    <TemplateList templates={sortedTemplates} onActionSelect={handleAction} />
                                ) : (
                                    <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
                                        {sortedTemplates.map((template) => (
                                            <TemplateCard key={template.id} template={template} onActionSelect={handleAction} />
                                        ))}
                                    </div>
                                )
                            ) : (
                                <div className="text-center p-8 text-muted-foreground">No templates found.</div>
                            )}
                        </>
                    )}
                </CardContent>
            </Card>

            <AlertDialog open={isAlertOpen} onOpenChange={setIsAlertOpen}>
                <AlertDialogContent><AlertDialogHeader><AlertDialogTitle>Are you sure?</AlertDialogTitle><AlertDialogDescription>This will permanently delete "{templateToDelete?.name}".</AlertDialogDescription></AlertDialogHeader><AlertDialogFooter><AlertDialogCancel onClick={() => setTemplateToDelete(null)}>Cancel</AlertDialogCancel><AlertDialogAction onClick={handleDeleteConfirm}>Delete</AlertDialogAction></AlertDialogFooter></AlertDialogContent>
            </AlertDialog>
            <EditTemplateDialog template={templateToEdit} open={isEditOpen} onOpenChange={setIsEditOpen} onSave={handleUpdateConfirm} />
            <ManualCreateDialog open={isManualCreateOpen} onOpenChange={setIsManualCreateOpen} onSave={handleManualCreateConfirm} />
        </>
    );
}

