'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import type { Template } from '@/lib/types';
import { getUserId } from '@/lib/auth';

/**
 * --- THIS IS THE DEFINITIVE FIX ---
 * This new utility function performs the exact 2-step cleaning process you described.
 * It is server-safe and robustly handles the corrupted data from your database.
 */
function cleanAndUnescapeHtml(rawDbBody: string): string {
    if (!rawDbBody) return '';

    // Step 1: Strip out the wrapping <div> tags from each line.
    const stripped = rawDbBody.replace(/<\/?div>/g, '');

    // Step 2: Unescape the HTML entities (e.g., &lt; becomes <).
    // This is a more robust, server-safe method than using DOMParser.
    return stripped
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"')
        .replace(/&#39;/g, "'")
        .replace(/&amp;/g, '&');
}


/**
 * Fetches all templates for the current user and cleans the body content.
 */
export async function getTemplates(): Promise<Template[]> {
  const userId = await getUserId();
  if (!userId) return [];
  try {
    const { rows } = await pool.query(
      'SELECT * FROM templates WHERE user_id = $1 ORDER BY last_modified DESC, created_at DESC',
      [userId]
    );
    // Apply the cleaning function to each template's body
    return rows.map(template => ({
        ...template,
        body: cleanAndUnescapeHtml(template.body)
    })) as Template[];
  } catch (error) {
    console.error('Database Error: Failed to fetch templates:', error);
    return [];
  }
}

/**
 * Fetches a single template by its ID and cleans the body content.
 */
export async function getTemplateById(templateId: number): Promise<Template | null> {
    const userId = await getUserId();
    if (!userId) return null;
    try {
        const { rows } = await pool.query(
            'SELECT * FROM templates WHERE id = $1 AND user_id = $2',
            [templateId, userId]
        );
        if (rows.length === 0) return null;

        const template = rows[0];
        return {
            ...template,
            body: cleanAndUnescapeHtml(template.body) // Clean the content here too
        } as Template;

    } catch (error) {
        console.error('Database Error: Failed to fetch template:', error);
        return null;
    }
}

/**
 * Adds a new template to the database. It saves the raw HTML from the editor.
 */
export async function addTemplate(templateData: Omit<Template, 'id' | 'created_at' | 'last_modified' | 'user_id'>) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: "Authentication failed." };

  const { name, subject, body } = templateData;
  try {
    const { rows } = await pool.query(
      `INSERT INTO templates (user_id, name, subject, body, created_at, last_modified)
       VALUES ($1, $2, $3, $4, NOW(), NOW())
       RETURNING *`,
      [userId, name, subject, body]
    );
    revalidatePath('/templates');
    return { success: true, message: 'Template created successfully!', template: rows[0] };
  } catch (error) {
    console.error('Database Error: Failed to create template:', error);
    return { success: false, message: 'Failed to create template.' };
  }
}

/**
 * Updates an existing template.
 */
export async function updateTemplate(templateId: number, templateData: Partial<Omit<Template, 'id' | 'user_id'>>) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: "Authentication failed." };

  const { name, subject, body } = templateData;
  try {
    const { rowCount } = await pool.query(
      `UPDATE templates SET name = $1, subject = $2, body = $3, last_modified = NOW() WHERE id = $4 AND user_id = $5`,
      [name, subject, body, templateId, userId]
    );
    if (rowCount === 0) return { success: false, message: "Template not found or permission denied." };

    revalidatePath('/templates');
    revalidatePath(`/templates/edit/${templateId}`);
    return { success: true, message: 'Template updated successfully.' };
  } catch (error) {
    console.error('Database Error: Failed to update template:', error);
    return { success: false, message: 'Failed to update template.' };
  }
}

/**
 * Deletes a template.
 */
export async function deleteTemplate(templateId: number) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: "Authentication failed." };

  try {
    const { rowCount } = await pool.query(
      'DELETE FROM templates WHERE id = $1 AND user_id = $2',
      [templateId, userId]
    );
    if (rowCount === 0) return { success: false, message: "Template not found or permission denied." };

    revalidatePath('/templates');
    return { success: true, message: 'Template deleted successfully.' };
  } catch (error) {
    console.error('Database Error: Failed to delete template:', error);
    return { success: false, message: 'Failed to delete template.' };
  }
}

