
'use client';

import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Filter, Inbox, Send, Sparkles, User, Search, Bot, Loader2 } from "lucide-react";
import { useState } from "react";
import { cn } from "@/lib/utils";
import { useToast } from "@/hooks/use-toast";
import { generateReplyDraft } from "@/ai/flows/generate-reply-draft";

type Reply = { 
    id: number; 
    from: string; 
    email: string; 
    subject: string; 
    snippet: string; 
    timestamp: string; 
    avatar: string;
    conversation: {
        sender: 'user' | 'prospect';
        subject?: string;
        body: string;
    }[];
};

const initialReplies: Reply[] = [
    { 
        id: 1, 
        from: 'Alice Johnson', 
        email: 'alice.j@corp.com', 
        subject: 'Re: Q2 Product Launch', 
        snippet: 'This looks interesting! Could you tell me more about the enterprise pricing?', 
        timestamp: '2 hours ago', 
        avatar: 'AJ',
        conversation: [
            { sender: 'user', subject: 'Q2 Product Launch', body: 'Hi Alice, I wanted to follow up on our new product. It\'s designed to streamline your workflow.' },
            { sender: 'prospect', body: 'This looks interesting! Could you tell me more about the enterprise pricing?' }
        ]
    },
    { 
        id: 2, 
        from: 'Out of Office', 
        email: 'autoreply@example.com', 
        subject: 'Automatic reply: Away until June 5th', 
        snippet: 'I am currently out of the office with limited access to email...', 
        timestamp: '5 hours ago', 
        avatar: 'OO',
        conversation: [
            { sender: 'user', subject: 'Quick Question', body: 'Hello, just wanted to ask...' },
            { sender: 'prospect', body: 'I am currently out of the office with limited access to email. I will respond to your message upon my return on June 5th.' }
        ]
    },
    { 
        id: 3, 
        from: 'Bob Williams', 
        email: 'bob.w@startup.io', 
        subject: 'Re: Following up', 
        snippet: 'Not the right time for us, but maybe check back in Q4. Thanks.', 
        timestamp: '1 day ago', 
        avatar: 'BW',
        conversation: [
            { sender: 'user', subject: 'Following up', body: 'Hi Bob, Checking in to see if you had any thoughts on our proposal.' },
            { sender: 'prospect', body: 'Not the right time for us, but maybe check back in Q4. Thanks.' }
        ]
    },
    { 
        id: 4, 
        from: 'Charlie Brown', 
        email: 'charlie@bigcompany.com', 
        subject: 'Question about your service', 
        snippet: 'Do you integrate with Salesforce?', 
        timestamp: '2 days ago', 
        avatar: 'CB',
        conversation: [
            { sender: 'user', subject: 'Your Integration Capabilities', body: 'Hi Charlie, Our platform offers robust integration options...' },
            { sender: 'prospect', body: 'Do you integrate with Salesforce?' }
        ]
    },
];

export default function SmartReplyInboxPage() {
    const { toast } = useToast();
    const [replies, setReplies] = useState<Reply[]>(initialReplies);
    const [selectedReply, setSelectedReply] = useState<Reply>(replies[0]);
    const [replyText, setReplyText] = useState('');
    const [isGenerating, setIsGenerating] = useState(false);

    const handleGenerateReply = async () => {
        setIsGenerating(true);
        setReplyText('');
        try {
            const lastProspectMessage = selectedReply.conversation.filter(c => c.sender === 'prospect').pop();
            if (!lastProspectMessage) {
                 toast({ variant: 'destructive', title: 'Error', description: 'Could not find a message to reply to.' });
                 return;
            }

            const result = await generateReplyDraft({
                prospectName: selectedReply.from,
                prospectMessage: lastProspectMessage.body,
                threadContext: selectedReply.conversation.map(m => `${m.sender === 'user' ? 'Me' : selectedReply.from}: ${m.body}`).join('\n')
            });

            setReplyText(result.replyDraft);
        } catch (error) {
            console.error(error);
            toast({ variant: 'destructive', title: 'AI Generation Failed', description: 'Could not generate a reply draft.' });
        } finally {
            setIsGenerating(false);
        }
    };
    
    const handleSend = () => {
        if (!replyText.trim()) {
            toast({ variant: 'destructive', title: 'Empty Reply', description: 'Cannot send an empty reply.' });
            return;
        }

        const newConversationEntry = {
            sender: 'user' as const,
            body: replyText
        };
        
        // Update the conversation for the selected reply
        const updatedConversation = [...selectedReply.conversation, newConversationEntry];
        const updatedSelectedReply = { 
            ...selectedReply, 
            conversation: updatedConversation,
            snippet: `You: ${replyText}`,
            timestamp: 'Just now'
        };

        // Update the main replies list
        const updatedReplies = replies.map(r => 
            r.id === selectedReply.id ? updatedSelectedReply : r
        );
        
        setReplies(updatedReplies);
        setSelectedReply(updatedSelectedReply);

        toast({ title: 'Reply Sent!', description: `Your reply to ${selectedReply.from} has been sent.` });
        setReplyText('');
    };

    return (
        <div className="h-[calc(100vh-8rem)] flex flex-col">
            <PageHeader
                title="Smart Reply Inbox"
                description="Manage campaign replies with AI-powered filtering and suggestions."
            >
                 <Button variant="outline">
                    <Filter className="mr-2 h-4 w-4" />
                    Filter Replies
                </Button>
            </PageHeader>
            <div className="grid lg:grid-cols-3 gap-8 items-start flex-1 min-h-0">
                {/* Left Column: Inbox List */}
                <Card className="lg:col-span-1 h-full flex flex-col">
                    <CardHeader className="flex-shrink-0">
                        <CardTitle className="flex items-center gap-2"><Inbox /> All Replies</CardTitle>
                        <div className="relative pt-2">
                             <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                             <Input placeholder="Search replies..." className="pl-10"/>
                        </div>
                    </CardHeader>
                    <div className="flex-1 overflow-y-auto border-t">
                        {replies.map(reply => (
                            <div 
                                key={reply.id} 
                                className={cn(
                                    "p-4 border-b cursor-pointer hover:bg-muted/50",
                                    selectedReply.id === reply.id && "bg-muted/80"
                                )}
                                onClick={() => {
                                    setSelectedReply(reply);
                                    setReplyText('');
                                }}
                            >
                                <div className="flex items-center justify-between mb-1">
                                    <p className="font-semibold">{reply.from}</p>
                                    <p className="text-xs text-muted-foreground">{reply.timestamp}</p>
                                </div>
                                <p className="text-sm text-muted-foreground truncate">{reply.subject}</p>
                                <p className="text-xs text-muted-foreground truncate">{reply.snippet}</p>
                            </div>
                        ))}
                    </div>
                </Card>

                {/* Right Column: Conversation & AI Tools */}
                <div className="lg:col-span-2 h-full flex flex-col gap-8">
                    <Card className="flex-1 flex flex-col min-h-0">
                        <CardHeader>
                             <div className="flex items-center gap-4">
                                <Avatar className="h-10 w-10">
                                    <AvatarFallback>{selectedReply.avatar}</AvatarFallback>
                                </Avatar>
                                <div>
                                    <CardTitle>{selectedReply.from}</CardTitle>
                                    <CardDescription>{selectedReply.email}</CardDescription>
                                </div>
                            </div>
                        </CardHeader>
                        <Separator />
                        <CardContent className="flex-1 py-6 space-y-4 overflow-y-auto">
                            {selectedReply.conversation.map((message, index) => (
                                <div key={index} className={cn("flex items-start gap-3", message.sender === 'user' && "justify-end")}>
                                    {message.sender === 'prospect' && (
                                        <Avatar className="w-8 h-8 border">
                                            <AvatarFallback>{selectedReply.avatar}</AvatarFallback>
                                        </Avatar>
                                    )}
                                    <div className={cn(
                                        "max-w-lg rounded-lg p-3 text-sm",
                                        message.sender === 'prospect' ? 'bg-muted' : 'bg-primary text-primary-foreground'
                                    )}>
                                        <p className="font-bold">{message.sender === 'user' ? 'You' : selectedReply.from}</p>
                                        {message.subject && <p className="text-xs opacity-80 mb-1">Subject: {message.subject}</p>}
                                        <p>{message.body}</p>
                                    </div>
                                    {message.sender === 'user' && (
                                        <Avatar className="w-8 h-8 border">
                                            <AvatarFallback><User size={18}/></AvatarFallback>
                                        </Avatar>
                                    )}
                                </div>
                            ))}
                        </CardContent>
                    </Card>
                    
                     <Card className="flex-shrink-0">
                        <CardContent className="p-4 space-y-4">
                            <div className="grid gap-2 relative">
                                <Textarea 
                                    placeholder="Type your reply here or use the AI assistant..." 
                                    rows={5}
                                    value={replyText}
                                    onChange={(e) => setReplyText(e.target.value)}
                                    className="pr-12"
                                />
                                <Button 
                                    variant="ghost" 
                                    size="icon" 
                                    className="absolute bottom-2 right-2 h-8 w-8"
                                    onClick={handleGenerateReply}
                                    disabled={isGenerating}
                                >
                                    {isGenerating ? <Loader2 className="animate-spin" /> : <Sparkles className="text-accent" />}
                                    <span className="sr-only">Generate with AI</span>
                                </Button>
                            </div>
                            <div className="flex gap-2">
                                <Button className="w-full" onClick={handleSend} disabled={isGenerating}>
                                    <Send className="mr-2 h-4 w-4" /> Send Reply
                                </Button>
                                <Button variant="outline" className="w-full" onClick={handleGenerateReply} disabled={isGenerating}>
                                    <Bot className="mr-2 h-4 w-4" /> Regenerate
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </div>
    );
}
