'use client';

import { useEffect, useRef, useState, useTransition, useActionState } from 'react'; // Corrected import
import { useFormStatus } from 'react-dom';
import {
  saveGmailConnection,
  deleteGmailConnection,
  createCustomSender,
  deleteCustomSender,
  verifyCustomSender
} from './actions';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Textarea } from '@/components/ui/textarea';
import { toast } from '@/hooks/use-toast'; // Assuming you use shadcn/ui toast

// --- TYPE DEFINITIONS (should ideally live in a types.ts file) ---
interface SmtpService {
  id: number;
  name: string;
}
interface CustomSender {
  id: number;
  domain_name: string;
  status: string;
  smtp_service: { name: string; smtp_host: string; };
}

interface SendingSettingsFormProps {
  initialGmailSenders: SmtpService[];
  initialPlatformSenders: SmtpService[];
  initialCustomSenders: CustomSender[];
}

// --- SUBMIT BUTTON COMPONENT ---
// A helper component to automatically show a pending state during form submission.
function SubmitButton({ children, variant = "default" }: { children: React.ReactNode, variant?: any }) {
  const { pending } = useFormStatus();
  return (
    <Button type="submit" disabled={pending} variant={variant} className="w-full">
      {pending ? 'Saving...' : children}
    </Button>
  );
}

// --- MAIN CLIENT COMPONENT ---
export function SendingSettingsForm({
  initialGmailSenders,
  initialPlatformSenders,
  initialCustomSenders,
}: SendingSettingsFormProps) {

  // State for optimistic updates and transitions
  const [isPending, startTransition] = useTransition();
  const [verifyingId, setVerifyingId] = useState<number | null>(null);

  // Form state management with useActionState hook - CORRECTED HOOK
  const [gmailFormState, gmailFormAction] = useActionState(saveGmailConnection, { message: '' });
  const [customSenderFormState, customSenderFormAction] = useActionState(createCustomSender, { message: '' });

  // Refs to reset forms after successful submission
  const gmailFormRef = useRef<HTMLFormElement>(null);
  const customSenderFormRef = useRef<HTMLFormElement>(null);

  // Effect to show toast messages and reset forms
  useEffect(() => {
    if (gmailFormState.message) {
      if (gmailFormState.success) {
        toast({ title: "Success", description: gmailFormState.message });
        gmailFormRef.current?.reset();
      } else {
        toast({ title: "Error", description: gmailFormState.message, variant: "destructive" });
      }
    }
  }, [gmailFormState]);

  useEffect(() => {
    if (customSenderFormState.message) {
       if (customSenderFormState.success) {
        toast({ title: "Success", description: customSenderFormState.message });
        customSenderFormRef.current?.reset();
      } else {
        toast({ title: "Error", description: customSenderFormState.message, variant: "destructive" });
      }
    }
  }, [customSenderFormState]);


  // --- HANDLER FUNCTIONS FOR ACTIONS THAT ARE NOT FORM SUBMISSIONS ---
  const handleDelete = (id: number, deleteAction: (id: number) => Promise<{message: string}>) => {
    if (confirm("Are you sure you want to delete this?")) {
      startTransition(async () => {
        const result = await deleteAction(id);
        toast({ title: "Action Result", description: result.message });
      });
    }
  };

  const handleVerify = (id: number) => {
    setVerifyingId(id);
    startTransition(async () => {
      const result = await verifyCustomSender(id);
      toast({ title: "Verification", description: result.message });
      setVerifyingId(null);
    });
  };

  return (
    <div className="grid gap-8 lg:grid-cols-2">
      <div className="space-y-8">
        {/* Section 1: Gmail Connections */}
        <Card>
          <CardHeader>
            <CardTitle>Add a Gmail / Google Workspace Account</CardTitle>
            <CardDescription>Enter your Gmail address and a 16-character App Password to connect.</CardDescription>
          </CardHeader>
          <CardContent>
            <form ref={gmailFormRef} action={gmailFormAction} className="space-y-4">
              <div>
                <Label htmlFor="gmailEmail">Gmail Address</Label>
                <Input id="gmailEmail" name="gmailEmail" type="email" placeholder="your.email@gmail.com" required />
              </div>
              <div>
                <Label htmlFor="googleAppPassword">Google App Password</Label>
                <Input id="googleAppPassword" name="googleAppPassword" type="password" placeholder="16-character App Password" required />
              </div>
              <SubmitButton>Save Gmail Connection</SubmitButton>
            </form>
            <div className="divide-y mt-6 pt-4 border-t">
              <h4 className="text-md font-semibold mb-2">Your Connected Gmail Accounts</h4>
              {initialGmailSenders.length > 0 ? (
                initialGmailSenders.map((service) => (
                  <div key={service.id} className="py-3 flex justify-between items-center">
                    <p className="font-medium">{service.name}</p>
                    <Button onClick={() => handleDelete(service.id, deleteGmailConnection)} variant="destructive" size="sm" disabled={isPending}>Delete</Button>
                  </div>
                ))
              ) : ( <p className="py-4 text-center text-sm text-muted-foreground">No Gmail connections added.</p> )}
            </div>
          </CardContent>
        </Card>

        {/* Platform Senders Section */}
        <Card>
          <CardHeader>
            <CardTitle>Platform Senders</CardTitle>
            <CardDescription>Pre-configured, high-deliverability sending services.</CardDescription>
          </CardHeader>
          <CardContent className="divide-y">
            {initialPlatformSenders.length > 0 ? (
              initialPlatformSenders.map((sender) => (
                <div key={sender.id} className="py-4 flex justify-between items-center">
                  <p className="font-medium">{sender.name}</p>
                  <span className="px-3 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">Verified & Active</span>
                </div>
              ))
            ) : ( <p className="py-4 text-center text-muted-foreground">No platform senders available.</p> )}
          </CardContent>
        </Card>
      </div>

      {/* Section 2: Custom Senders & Domains */}
      <Card>
        <CardHeader>
          <CardTitle>Add New Custom Sender & Domain</CardTitle>
        </CardHeader>
        <CardContent>
          {/* --- THIS IS THE CORRECTED FORM --- */}
          <form ref={customSenderFormRef} action={customSenderFormAction} className="space-y-6">
            <fieldset className="border p-4 rounded-md">
                <legend className="text-lg font-medium px-2">Sender Identity</legend>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 pt-2">
            {/* --- ADD THESE TWO NEW FIELDS --- */}
            <div>
                <Label htmlFor="fromName">"From" Name</Label>
                <Input id="fromName" name="fromName" placeholder="e.g., John from Company" required />
                <p className="text-xs text-muted-foreground pt-1">The name recipients will see.</p>
            </div>
            <div>
                <Label htmlFor="fromEmail">"From" Email Address</Label>
                <Input id="fromEmail" name="fromEmail" type="email" placeholder="e.g., john@yourdomain.com" required />
                <p className="text-xs text-muted-foreground pt-1">The email recipients will reply to.</p>
            </div>
          {/* --- END OF NEW FIELDS --- */}
          </div>
      </fieldset>
            <fieldset className="border p-4 rounded-md">
              <legend className="text-lg font-medium px-2">SMTP Settings</legend>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 pt-2">
                <div>
                  <Label htmlFor="name">Connection Name</Label>
                  {/* name="name" is correct */}
                  <Input id="name" name="name" placeholder="My SendGrid Account" required />
                </div>
                <div>
                  <Label htmlFor="smtpHost">SMTP Host</Label>
                  {/* CORRECTED: name="smtpHost" */}
                  <Input id="smtpHost" name="smtpHost" placeholder="e.g., smtp.sendgrid.net" required />
                </div>
                <div>
                  <Label htmlFor="smtpPort">SMTP Port</Label>
                  {/* CORRECTED: name="smtpPort" */}
                  <Input id="smtpPort" name="smtpPort" type="number" placeholder="587" required />
                </div>
                <div>
                  <Label htmlFor="smtpUsername">Username</Label>
                  {/* CORRECTED: name="smtpUsername" */}
                  <Input id="smtpUsername" name="smtpUsername" placeholder="e.g., apikey" required />
                </div>
                <div>
                  <Label htmlFor="smtpPassword">Password / API Key</Label>
                  {/* CORRECTED: name="smtpPassword" */}
                  <Input id="smtpPassword" name="smtpPassword" type="password" required />
                </div>
              </div>
            </fieldset>
            <fieldset className="border p-4 rounded-md">
              <legend className="text-lg font-medium px-2">Domain & DNS Records</legend>
              <div className="space-y-4 pt-2">
                <div>
                  <Label htmlFor="domainName">Sending Domain</Label>
                  {/* name="domainName" is correct */}
                  <Input id="domainName" name="domainName" placeholder="mycompany.com" required />
                </div>
                <div className="space-y-2">
                  <Label>DNS Records (from your SMTP provider)</Label>
                  <div className="grid grid-cols-1 gap-4">
                    <div>
                      <Label htmlFor="spfHost" className="text-xs font-semibold">SPF Host:</Label>
                      {/* CORRECTED: name="spfHost" */}
                      <Input id="spfHost" name="spfHost" defaultValue="@" required />
                    </div>
                    <div>
                      <Label htmlFor="spfValue" className="text-xs font-semibold">SPF Value:</Label>
                      {/* CORRECTED: name="spfValue" */}
                      <Textarea id="spfValue" name="spfValue" placeholder="v=spf1 include:sendgrid.net ~all" required />
                    </div>
                     <div>
                      <Label htmlFor="dkimHost" className="text-xs font-semibold">DKIM Host:</Label>
                      {/* CORRECTED: name="dkimHost" */}
                      <Input id="dkimHost" name="dkimHost" defaultValue="dkim._domainkey" required />
                    </div>
                    <div>
                      <Label htmlFor="dkimValue" className="text-xs font-semibold">DKIM Value:</Label>
                      {/* CORRECTED: name="dkimValue" */}
                      <Textarea id="dkimValue" name="dkimValue" placeholder="v=DKIM1; k=rsa; p=..." required />
                    </div>
                     <div>
                      <Label htmlFor="dmarcHost" className="text-xs font-semibold">DMARC Host:</Label>
                      {/* CORRECTED: name="dmarcHost" */}
                      <Input id="dmarcHost" name="dmarcHost" defaultValue="_dmarc" required />
                    </div>
                    <div>
                      <Label htmlFor="dmarcValue" className="text-xs font-semibold">DMARC Value:</Label>
                       {/* CORRECTED: name="dmarcValue" */}
                      <Textarea id="dmarcValue" name="dmarcValue" placeholder="v=DMARC1; p=none;" required />
                    </div>
                  </div>
                </div>
              </div>
            </fieldset>
            <SubmitButton variant="secondary">Save Custom Sender</SubmitButton>
          </form>
          {/* --- END OF CORRECTED FORM --- */}
          <div className="divide-y mt-6 pt-4 border-t">
            <h4 className="text-md font-semibold mb-2">Your Saved Custom Senders</h4>
            {initialCustomSenders.length > 0 ? (
              initialCustomSenders.map((sender) => (
                <div key={sender.id} className="py-3 flex justify-between items-center gap-2">
                  <div>
                    <p className="font-semibold">{sender.smtp_service.name}</p>
                    <p className="text-sm text-muted-foreground">{sender.domain_name}</p>
                  </div>
                  <div className="flex items-center space-x-2 flex-shrink-0">
                    <span className={`px-3 py-1 text-xs font-semibold rounded-full ${sender.status === 'Verified' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}`}>
                      {sender.status}
                    </span>
                    <Button onClick={() => handleVerify(sender.id)} size="sm" disabled={isPending || verifyingId === sender.id}>
                      {verifyingId === sender.id ? 'Verifying...' : 'Verify'}
                    </Button>
                    <Button onClick={() => handleDelete(sender.id, deleteCustomSender)} variant="destructive" size="sm" disabled={isPending}>Delete</Button>
                  </div>
                </div>
              ))
            ) : ( <p className="py-4 text-center text-sm text-muted-foreground">No custom senders added.</p> )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
