'use server';

import { revalidatePath } from 'next/cache';
import { z } from 'zod';
// Make sure you have a configured node-postgres pool client
// This is a common way to set it up in a Next.js project
import { pool } from '@/lib/db';

/**
 * Fetches all sender data from the actual database.
 */
export async function getSenderData() {
  try {
    // Fetch all services (gmail, platform, etc.)
    const servicesRes = await pool.query('SELECT id, name, service_type FROM smtp_services');

    // Fetch custom senders and JOIN with smtp_services to get the name and host
    const customSendersRes = await pool.query(`
      SELECT
        cs.id,
        cs.domain_name,
        cs.status,
        ss.name as service_name,
        ss.smtp_host as service_host
      FROM custom_senders cs
      JOIN smtp_services ss ON cs.smtp_service_id = ss.id
    `);

    const allServices = servicesRes.rows;

    // The data structure for the frontend needs to be assembled
    const customSenders = customSendersRes.rows.map(row => ({
        id: row.id,
        domain_name: row.domain_name,
        status: row.status,
        smtp_service: {
            name: row.service_name,
            smtp_host: row.service_host
        }
    }));

    return {
      gmailSenders: allServices.filter(s => s.service_type === 'gmail'),
      platformSenders: allServices.filter(s => s.service_type === 'platform'),
      customSenders: customSenders,
    };
  } catch (error) {
    console.error("Database Error: Failed to fetch sender data.", error);
    return {
        gmailSenders: [],
        platformSenders: [],
        customSenders: [],
        error: "Could not load data from the database."
    };
  }
}

/**
 * Saves a new Gmail connection to the smtp_services table.
 */
export async function saveGmailConnection(prevState: any, formData: FormData) {
  const schema = z.object({
    email: z.string().email("Please enter a valid email address."),
    appPassword: z.string().min(1, "App Password cannot be empty."), // Google App Passwords are 16 chars, but we won't enforce length here.
  });

  const validatedFields = schema.safeParse({
    email: formData.get('gmailEmail'),
    appPassword: formData.get('googleAppPassword'),
  });

  if (!validatedFields.success) {
    return { message: 'Invalid form data.', errors: validatedFields.error.flatten().fieldErrors };
  }

  const { email, appPassword } = validatedFields.data;

  try {
    await pool.query(
      `INSERT INTO smtp_services (name, service_type, smtp_host, smtp_port, smtp_username, smtp_password)
       VALUES ($1, 'gmail', 'smtp.gmail.com', 587, $1, $2)`,
      [email, appPassword] // Using the email as the username for Gmail
    );

    revalidatePath('/settings/sending-settings');
    return { message: 'Gmail connection saved successfully!', success: true };
  } catch (error) {
    console.error("Database Error: Failed to save Gmail connection.", error);
    return { message: 'Failed to save Gmail connection. The email might already be in use.' };
  }
}

/**
 * Creates a new custom sender. This involves a transaction to insert into
 * both smtp_services and custom_senders tables safely.
 */
export async function createCustomSender(prevState: any, formData: FormData) {
    // 1. --- UPDATE THE ZOD SCHEMA HERE ---
    // We've added fromName and fromEmail for validation.
    const schema = z.object({
        name: z.string().min(1, "Connection name is required."),
        smtpHost: z.string().min(1, "SMTP Host is required."),
        smtpPort: z.coerce.number().int().positive("SMTP Port must be a positive number."),
        smtpUsername: z.string().min(1, "Username is required."),
        smtpPassword: z.string().min(1, "Password is required."),
        domainName: z.string().min(1, "Domain name is required."),
        fromName: z.string().min(1, '"From" name is required.'),   // <-- ADDED THIS
        fromEmail: z.string().email('"From" email must be a valid email.'), // <-- ADDED THIS
        spfHost: z.string().min(1, "SPF Host is required."),
        spfValue: z.string().min(1, "SPF Value is required."),
        dkimHost: z.string().min(1, "DKIM Host is required."),
        dkimValue: z.string().min(1, "DKIM Value is required."),
        dmarcHost: z.string().min(1, "DMARC Host is required."),
        dmarcValue: z.string().min(1, "DMARC Value is required."),
    });

    const validatedFields = schema.safeParse(Object.fromEntries(formData.entries()));

    if (!validatedFields.success) {
        // Log the detailed validation errors for easier debugging
        console.error("Form validation failed:", validatedFields.error.flatten().fieldErrors);
        return { message: 'Invalid form data. Please check all fields.', success: false };
    }

    // 2. --- DESTRUCTURE THE NEW FIELDS ---
    const data = validatedFields.data;
    const client = await pool.connect();

    try {
        await client.query('BEGIN');

        // 3. --- UPDATE THE SQL INSERT QUERY ---
        // Add from_name and from_email to the query and its parameters.
        const serviceInsertQuery = `
            INSERT INTO smtp_services (name, service_type, smtp_host, smtp_port, smtp_username, smtp_password, from_name, from_email)
            VALUES ($1, 'custom', $2, $3, $4, $5, $6, $7)
            RETURNING id
        `;
        const serviceResult = await client.query(serviceInsertQuery, [
            data.name, data.smtpHost, data.smtpPort, data.smtpUsername, data.smtpPassword,
            data.fromName, data.fromEmail // <-- ADDED THE NEW VARIABLES HERE
        ]);
        const newServiceId = serviceResult.rows[0].id;

        const senderInsertQuery = `
            INSERT INTO custom_senders (smtp_service_id, domain_name, spf_host, spf_value, dkim_host, dkim_value, dmarc_host, dmarc_value)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        `;
        await client.query(senderInsertQuery, [newServiceId, data.domainName, data.spfHost, data.spfValue, data.dkimHost, data.dkimValue, data.dmarcHost, data.dmarcValue]);

        await client.query('COMMIT');

        revalidatePath('/settings/sending-settings');
        return { message: 'Custom sender created successfully!', success: true };
    } catch (error) {
        await client.query('ROLLBACK');
        console.error("Database Error: Failed to create custom sender.", error);
        return { message: 'Failed to create custom sender. The domain or connection name may already exist.', success: false };
    } finally {
        client.release();
    }
}

/**
 * Deletes a service. ON DELETE CASCADE in the database will also delete the linked custom_sender.
 */
export async function deleteGmailConnection(serviceId: number) {
    try {
        await pool.query('DELETE FROM smtp_services WHERE id = $1 AND service_type = \'gmail\'', [serviceId]);
        revalidatePath('/settings/sending-settings');
        return { message: 'Gmail connection deleted successfully.' };
    } catch (error) {
        console.error("Database Error: Failed to delete Gmail connection.", error);
        return { message: 'Failed to delete connection.' };
    }
}

/**
 * Deletes a custom sender. This will cascade and delete the associated smtp_service.
 */
export async function deleteCustomSender(senderId: number) {
    try {
        // We need to get the smtp_service_id from the custom_senders table first
        const res = await pool.query('SELECT smtp_service_id FROM custom_senders WHERE id = $1', [senderId]);
        if (res.rows.length === 0) {
            return { message: 'Sender not found.' };
        }
        const serviceIdToDelete = res.rows[0].smtp_service_id;

        // Deleting the service will cascade and delete the custom_sender row
        await pool.query('DELETE FROM smtp_services WHERE id = $1', [serviceIdToDelete]);

        revalidatePath('/settings/sending-settings');
        return { message: 'Custom sender deleted successfully.' };
    } catch (error) {
        console.error("Database Error: Failed to delete custom sender.", error);
        return { message: 'Failed to delete sender.' };
    }
}

/**
 * Triggers the verification process for a custom sender's DNS records.
 */
export async function verifyCustomSender(senderId: number) {
    try {
        console.log(`SERVER ACTION: Verifying sender with ID: ${senderId}`);
        // TODO: Implement your actual DNS verification logic here.
        // This would involve using a library like 'dns' to perform TXT record lookups
        // for SPF, DKIM, and DMARC, and then updating the status in the database.

        // For now, we'll just simulate a successful verification.
        await pool.query(
            "UPDATE custom_senders SET status = 'Verified', verified_at = NOW() WHERE id = $1",
            [senderId]
        );

        revalidatePath('/settings/sending-settings');
        return { message: 'Sender successfully verified!' };
    } catch (error) {
        console.error("Verification Error:", error);
        await pool.query("UPDATE custom_senders SET status = 'Failed' WHERE id = $1", [senderId]);
        revalidatePath('/settings/sending-settings');
        return { message: 'Verification failed.' };
    }
}
