
'use client';

import { PageHeader } from "@/components/page-header";
import { Card, CardContent, CardHeader, CardTitle, CardDescription, CardFooter } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { KeyRound, Trash2, Upload, Save, CheckCircle, Loader2 } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import Link from "next/link";
import React, { useEffect, useState, useRef } from "react";
import { useToast } from "@/hooks/use-toast";
import { useRouter } from "next/navigation";
import { useUsers } from "@/context/UserContext";
import type { User } from "@/lib/types";
import { useActivityLog } from "@/context/ActivityLogContext";
import { getCurrentUser } from "@/app/(app)/admin/actions";

export default function SettingsPage() {
  const { toast } = useToast();
  const router = useRouter();
  const { updateUser, fetchUsers } = useUsers();
  const { addLog, fetchLogs } = useActivityLog();
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  const [userName, setUserName] = useState('');
  const [userEmail, setUserEmail] = useState('');
  const [profilePic, setProfilePic] = useState<string | null>(null);
  const [initialProfilePic, setInitialProfilePic] = useState<string | null>(null);
  const [profilePicFile, setProfilePicFile] = useState<File | null>(null);

  const [currentPassword, setCurrentPassword] = useState('');
  const [newPassword, setNewPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');

  const fileInputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    const fetchUser = async () => {
        if (typeof window !== 'undefined') {
            const email = sessionStorage.getItem('userEmail');
            if (email) {
                const user = await getCurrentUser(email);
                if (user) {
                    setCurrentUser(user);
                    setUserName(user.name);
                    setUserEmail(user.email);
                    // Set profile picture from database, fallback to sessionStorage
                    const dbProfilePic = user.profile_pic_url;
                    if (dbProfilePic) {
                        setProfilePic(dbProfilePic);
                        setInitialProfilePic(dbProfilePic);
                        // Update sessionStorage with database value
                        sessionStorage.setItem('profilePic', dbProfilePic);
                    } else {
                        // Fallback to sessionStorage if no database value
                        const sessionProfilePic = sessionStorage.getItem('profilePic');
                        setProfilePic(sessionProfilePic);
                        setInitialProfilePic(sessionProfilePic);
                    }
                } else {
                    // If no user found, still check sessionStorage for profile pic
                    const pic = sessionStorage.getItem('profilePic');
                    setProfilePic(pic);
                    setInitialProfilePic(pic);
                }
            }
        }
        setIsLoading(false);
    }
    fetchUser();
  }, []);

  const getInitials = (name?: string | null) => {
    if (!name) return 'U';
    return name
        .split(' ')
        .map(n => n[0])
        .slice(0, 2)
        .join('')
        .toUpperCase();
  }
  
  const handleSaveChanges = async () => {
    if (typeof window !== 'undefined' && currentUser) {
        let updatedUser: Partial<User> = {};
        let detailsChanged = false;
        const logDetails = [];

        // Name update logic
        if (userName !== currentUser.name) {
            logDetails.push(`Name changed from "${currentUser.name}" to "${userName}".`);
            updatedUser.name = userName;
            detailsChanged = true;
        }
        
        // Profile picture update logic
        if (profilePicFile) { // Check if a new file was selected
            logDetails.push('Profile picture uploaded.');
            detailsChanged = true;
            
            const formData = new FormData();
            formData.append('profilePicture', profilePicFile); // 'profilePicture' is the key your backend expects

            try {
                const response = await fetch('/api/upload-profile-picture', { // Replace with your actual backend upload URL
                    method: 'POST',
                    body: formData,
                });

                if (!response.ok) {
                    throw new Error('Profile picture upload failed.');
                }
                const data = await response.json();
                if (data.url) {
                    updatedUser.profile_pic_url = data.url; // Assuming backend returns { url: '...' }
                }
            } catch (error) {
                console.error('Error uploading profile picture:', error);
                toast({ variant: "destructive", title: "Upload Failed", description: "Could not upload profile picture. Please try again." });
                return; // Stop the process if upload fails
            }
        } else if (profilePic !== initialProfilePic) {
             // Handle case where profile pic is removed but no new one uploaded
             logDetails.push('Profile picture was updated.'); // Or 'Profile picture removed.'
             detailsChanged = true;
             updatedUser.profile_pic_url = profilePic;
        }

        // Password update logic - requires a separate server action for security
        if (currentPassword || newPassword || confirmPassword) {
            toast({ variant: "destructive", title: "Password Change Not Implemented", description: "Password changes must be handled by a dedicated secure endpoint. This feature is not yet connected." });
            return;
        }

        if (!detailsChanged) {
             toast({ title: "No Changes", description: "No changes were detected to save." });
             return;
        }
        
        await updateUser(currentUser.id, updatedUser);
        await fetchUsers(); // This re-fetches all users, useful for admin context

        await addLog({
            action: 'Account Modified',
            userId: currentUser.id,
            userName: updatedUser.name || currentUser.name,
            details: logDetails.join(' ')
        });
        await fetchLogs();


        // Update session storage for immediate UI feedback and login state
        sessionStorage.setItem('userName', userName);
        if (updatedUser.profile_pic_url) {
            sessionStorage.setItem('profilePic', updatedUser.profile_pic_url);
        } else if (profilePic) {
            try {
                sessionStorage.setItem('profilePic', profilePic);
            } catch (error) {
                 toast({ variant: "destructive", title: "Could not save profile picture", description: "The image is too large. Please choose a smaller file." });
                 setProfilePic(initialProfilePic);
                 return;
            }
        } else {
            sessionStorage.removeItem('profilePic');
        }

        window.dispatchEvent(new Event('sessionStorageChanged'));

        toast({ title: "Settings Saved", description: "Your account details have been updated." });

        if (currentUser.role === 'Admin') {
            router.push('/admin');
        } else {
            router.push('/campaigns');
        }
        
        setInitialProfilePic(profilePic); 
    }
  };


  const handleProfilePicUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      setProfilePicFile(file); // Store the file object
      reader.onload = (event) => {
        const img = document.createElement('img');
        img.onload = () => {
          const canvas = document.createElement('canvas');
          const MAX_WIDTH = 100;
          const MAX_HEIGHT = 100;
          let width = img.width;
          let height = img.height;

          if (width > height) {
            if (width > MAX_WIDTH) {
              height *= MAX_WIDTH / width;
              width = MAX_WIDTH;
            }
          } else {
            if (height > MAX_HEIGHT) {
              width *= MAX_HEIGHT / height;
              height = MAX_HEIGHT;
            }
          }
          canvas.width = width;
          canvas.height = height;
          const ctx = canvas.getContext('2d');
          if (ctx) {
            ctx.drawImage(img, 0, 0, width, height);
            const dataUrl = canvas.toDataURL(file.type);
            setProfilePic(dataUrl);
          }
        };
        if (event.target?.result) {
            img.src = event.target.result as string;
        }
      };
      reader.readAsDataURL(file);
    }
  };

  const handleRemoveProfilePic = () => {
    setProfilePic(null);
    setProfilePicFile(null); // Also clear the file state
  };
  
  const handleUploadClick = () => {
      // Clear the file input value to allow selecting the same file again
      if (fileInputRef.current) {
          fileInputRef.current.value = '';
      }
      fileInputRef.current?.click();
  }

  if (isLoading) {
      return (
        <div className="flex justify-center items-center h-64">
            <Loader2 className="w-8 h-8 animate-spin text-primary" />
        </div>
      );
  }

  if (!currentUser) {
      return (
        <Card>
            <CardContent className="p-6 text-center text-muted-foreground">
                Could not load user data. Please try logging in again.
            </CardContent>
        </Card>
      );
  }

  return (
    <>
      <h1 className="text-3xl font-bold">Settings</h1>
      <div className="grid gap-8">
        <Card>
          <CardHeader>
            <CardTitle>Personal Information</CardTitle>
            <CardDescription>Update your full name and view your registered email.</CardDescription>
          </CardHeader>
          <CardContent className="grid gap-6">
            <div className="grid gap-2">
              <Label htmlFor="full-name">Full Name</Label>
              <Input id="full-name" value={userName} onChange={(e) => setUserName(e.target.value)} />
            </div>
            <div className="grid gap-2">
              <Label htmlFor="email">Email</Label>
              <div className="flex items-center gap-2">
                 <Input id="email" value={userEmail} disabled />
                 <Badge variant="outline" className="text-green-600 border-green-600/50 flex-shrink-0">
                    <CheckCircle className="mr-1 h-3 w-3" />
                    Verified
                 </Badge>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Change Password</CardTitle>
            <CardDescription>Update your password here. Leave blank to keep the same.</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4">
              <div className="grid gap-2">
                <Label htmlFor="current-password">Current Password</Label>
                <Input id="current-password" type="password" value={currentPassword} onChange={(e) => setCurrentPassword(e.target.value)} />
              </div>
              <div className="grid sm:grid-cols-2 gap-4">
                 <div className="grid gap-2">
                    <Label htmlFor="new-password">New Password</Label>
                    <Input id="new-password" type="password" value={newPassword} onChange={(e) => setNewPassword(e.target.value)} />
                </div>
                <div className="grid gap-2">
                    <Label htmlFor="confirm-password">Confirm Password</Label>
                    <Input id="confirm-password" type="password" value={confirmPassword} onChange={(e) => setConfirmPassword(e.target.value)} />
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
            <CardHeader>
                <CardTitle>Profile Picture</CardTitle>
                <CardDescription>Update your avatar.</CardDescription>
            </CardHeader>
            <CardContent>
                <div className="flex items-center gap-4">
                    <Avatar className="h-20 w-20">
                        <AvatarImage src={profilePic || undefined} alt="User Avatar" />
                        <AvatarFallback>{getInitials(userName)}</AvatarFallback>
                    </Avatar>
                    <div className="flex gap-2">
                        <Button variant="outline" onClick={handleUploadClick}>
                            <Upload className="mr-2 h-4 w-4" /> Change Photo
                        </Button>
                        <Button variant="ghost" onClick={handleRemoveProfilePic}>
                            <Trash2 className="mr-2 h-4 w-4" /> Remove
                        </Button>
                        <input
                            type="file"
                            ref={fileInputRef}
                            onChange={handleProfilePicUpload}
                            className="hidden"
                            accept="image/*"
                        />
                    </div>
                </div>
            </CardContent>
        </Card>
        
        <div className="flex justify-end">
            <Button onClick={handleSaveChanges}>
                <Save className="mr-2 h-4 w-4" />
                Save Changes
            </Button>
        </div>
      </div>
    </>
  );
}
