'use client';

import { useState, useTransition, useActionState, useEffect, useRef } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter, DialogClose } from '@/components/ui/dialog';
import { Badge } from '@/components/ui/badge';
import { PlusCircle, Trash2, RefreshCw, CheckCircle, XCircle, Clock } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { addTrackingDomain, deleteTrackingDomain, verifyDomain } from './actions';
import { cn } from '@/lib/utils';

type TrackingDomain = { id: number; domain_name: string; status: string; };

function AddDomainForm({ setOpen }: { setOpen: (open: boolean) => void }) {
  const [state, formAction] = useActionState(addTrackingDomain, { success: false, message: '' });
  const { toast } = useToast();
  const formRef = useRef<HTMLFormElement>(null);

  useEffect(() => {
    if (state.message) {
      toast({ title: state.success ? 'Success' : 'Error', description: state.message, variant: state.success ? 'default' : 'destructive' });
      if (state.success) {
        formRef.current?.reset();
        setOpen(false);
      }
    }
  }, [state, toast, setOpen]);

  return (
    <form action={formAction} ref={formRef}>
      <div className="grid gap-4 py-4">
        {/* --- UPGRADED DIALOG --- */}
        <div className="space-y-2">
          <Label htmlFor="domainName">Tracking Domain</Label>
          <Input id="domainName" name="domainName" placeholder="e.g., track.yourcompany.com" />
        </div>
        <div className="space-y-2">
          <Label htmlFor="cnameTarget">CNAME "Points to" Value</Label>
          <Input id="cnameTarget" name="cnameTarget" placeholder="e.g., your-app.vercel.app" />
        </div>
      </div>
      <DialogFooter>
        <DialogClose asChild><Button variant="ghost">Cancel</Button></DialogClose>
        <Button type="submit">Add Domain</Button>
      </DialogFooter>
    </form>
  );
}

export function DomainsDashboard({ initialDomains }: { initialDomains: TrackingDomain[] }) {
  const { toast } = useToast();
  const [isPending, startTransition] = useTransition();
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);

  const handleDelete = (id: number) => {
    if (confirm('Are you sure you want to delete this domain?')) {
      startTransition(async () => {
        const result = await deleteTrackingDomain(id);
        toast({ title: result.success ? 'Success' : 'Error', description: result.message, variant: result.success ? 'default' : 'destructive' });
      });
    }
  };

  const handleVerify = (domainName: string) => {
    startTransition(async () => {
      toast({ description: `Verifying ${domainName}...` });
      // The action now gets the target from the database, so we only need to send the domain name.
      const result = await verifyDomain(domainName);
      toast({ title: result.success ? 'Success' : 'Verification Failed', description: result.message, variant: result.success ? 'default' : 'destructive' });
    });
  };

  return (
    <div className="space-y-8">
      <Card>
        <CardHeader className="flex flex-row items-center justify-between">
          <div>
            <CardTitle>Your Tracking Domains</CardTitle>
            <CardDescription>Manage the domains used for tracking email opens and clicks.</CardDescription>
          </div>
          <Button onClick={() => setIsAddDialogOpen(true)}>
            <PlusCircle className="mr-2 h-4 w-4" /> Add New Domain
          </Button>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Domain Name</TableHead>
                <TableHead>Status</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {initialDomains.length > 0 ? (
                initialDomains.map((domain) => (
                  <TableRow key={domain.id}>
                    <TableCell className="font-mono">{domain.domain_name}</TableCell>
                    <TableCell>
                      <Badge variant="outline" className={cn({
                          'text-yellow-600 border-yellow-300': domain.status === 'Pending',
                          'text-green-600 border-green-300': domain.status === 'Verified',
                          'text-red-600 border-red-300': domain.status === 'Failed',
                      })}>
                        {domain.status === 'Pending' && <Clock className="mr-1 h-3 w-3" />}
                        {domain.status === 'Verified' && <CheckCircle className="mr-1 h-3 w-3" />}
                        {domain.status === 'Failed' && <XCircle className="mr-1 h-3 w-3" />}
                        {domain.status}
                      </Badge>
                    </TableCell>
                    <TableCell className="text-right space-x-2">
                      <Button variant="outline" size="sm" onClick={() => handleVerify(domain.domain_name)} disabled={isPending}>
                        <RefreshCw className="mr-2 h-4 w-4" /> Verify
                      </Button>
                      <Button variant="destructive" size="sm" onClick={() => handleDelete(domain.id)} disabled={isPending}>
                        <Trash2 className="mr-2 h-4 w-4" /> Delete
                      </Button>
                    </TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={3} className="text-center h-24">No tracking domains added yet.</TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Add a New Tracking Domain</DialogTitle>
            <DialogDescription>Enter the domain and the CNAME value it should point to.</DialogDescription>
          </DialogHeader>
          <AddDomainForm setOpen={setIsAddDialogOpen} />
        </DialogContent>
      </Dialog>
    </div>
  );
}
