

'use client';

import { useState, useMemo } from 'react';
import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardDescription } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Search, Eye, ChevronDown, Clock, ArrowUpAZ, ArrowDownAZ } from "lucide-react";
import Link from "next/link";
import { useCampaigns } from '@/context/CampaignContext';
import { useUsers } from '@/context/UserContext';
import type { Campaign } from '@/lib/types';
import { format } from 'date-fns';
import { Badge } from "@/components/ui/badge";
import { cn } from '@/lib/utils';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { useTemplates } from '@/context/TemplateContext';
import { Separator } from '@/components/ui/separator';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

type SortConfig = {
    key: keyof Campaign | 'recipient' | 'lastModified';
    direction: 'asc' | 'desc';
}

function ViewSentEmailDialog({ campaign, open, onOpenChange }: { campaign: Campaign | null, open: boolean, onOpenChange: (open: boolean) => void }) {
    const { getTemplate } = useTemplates();

    if (!campaign) return null;
    
    const isOneOff = campaign.audience.source === 'One-Off';
    const recipientName = isOneOff && campaign.audience.criteria.length > 0 ? campaign.audience.criteria[0]?.value.split('@')[0] : 'Audience';
    const templateId = campaign.sequence.steps[0]?.templateId;
    const template = templateId ? getTemplate(templateId) : null;
    
    const bodyText = template?.body ?? campaign.sequence.steps[0]?.description ?? 'No content found.';

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl">
                <DialogHeader>
                    <DialogTitle>Sent Email Details</DialogTitle>
                    <DialogDescription>
                        To: {isOneOff && campaign.audience.criteria[0] ? campaign.audience.criteria[0].value : campaign.name}
                    </DialogDescription>
                </DialogHeader>
                <div className="py-4 space-y-4 max-h-[70vh] overflow-y-auto">
                    <div className="space-y-1">
                        <h3 className="font-semibold">Subject: {template?.subject ?? 'No Subject'}</h3>
                         <p className="text-sm text-muted-foreground">Sent on: {format(new Date(campaign.last_modified), 'PPP p')}</p>
                    </div>
                    <Separator />
                    <div 
                        className="prose prose-sm dark:prose-invert max-w-none"
                        dangerouslySetInnerHTML={{ __html: bodyText.replace(/\n/g, '<br />') }}
                    />
                </div>
            </DialogContent>
        </Dialog>
    )
}

function SentEmailsTable({ logs, onActionClick }: { logs: any[], onActionClick: (log: Campaign) => void }) {
    return (
         <Table>
            <TableHeader>
                <TableRow>
                    <TableHead>User</TableHead>
                    <TableHead>Recipient/Campaign</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>Sent Date</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                </TableRow>
            </TableHeader>
            <TableBody>
                {logs.map(log => {
                    const isOneOff = log.audience.source === 'One-Off';
                    return (
                        <TableRow key={log.id}>
                            <TableCell className="font-medium">{log.userName}</TableCell>
                            <TableCell>{log.recipient}</TableCell>
                            <TableCell>
                                <Badge variant={isOneOff ? 'secondary' : 'default'} className={cn(isOneOff && 'bg-blue-100 text-blue-800')}>
                                    {isOneOff ? 'One-Off' : 'Campaign'}
                                </Badge>
                            </TableCell>
                            <TableCell>{format(new Date(log.last_modified), 'yyyy-MM-dd HH:mm:ss')}</TableCell>
                            <TableCell className="text-right">
                                    <Button variant="outline" size="sm" onClick={() => onActionClick(log)}>
                                    <Eye className="mr-2 h-4 w-4" />
                                    View
                                </Button>
                            </TableCell>
                        </TableRow>
                    );
                })}
                {logs.length === 0 && (
                    <TableRow>
                        <TableCell colSpan={5} className="h-24 text-center">
                            No sent emails found.
                        </TableCell>
                    </TableRow>
                )}
            </TableBody>
        </Table>
    )
}

export default function SentEmailsPage() {
    const { campaigns } = useCampaigns();
    const { users } = useUsers();
    const [searchTerm, setSearchTerm] = useState('');
    const [sortConfig, setSortConfig] = useState<SortConfig>({ key: 'lastModified', direction: 'desc' });
    const [selectedCampaign, setSelectedCampaign] = useState<Campaign | null>(null);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [activeTab, setActiveTab] = useState('all');

    const sentEmails = campaigns.filter(c => c.status === 'Sent');

    const mappedAndSortedLogs = useMemo(() => {
        const mapped = sentEmails.map(log => {
            const isOneOff = log.audience.source === 'One-Off';
            const recipient = isOneOff && log.audience.criteria.length > 0 && log.audience.criteria[0]?.value
                ? log.audience.criteria[0].value
                : log.name;
            const user = users.find(u => u.id === log.userId);
            return {
                ...log,
                recipient,
                userName: user ? user.name : 'Unknown User'
            }
        }).filter(log => 
             log.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
             (log.recipient && log.recipient.toLowerCase().includes(searchTerm.toLowerCase()))
        );

        return mapped.sort((a, b) => {
            const aVal = a[sortConfig.key as keyof typeof a];
            const bVal = b[sortConfig.key as keyof typeof b];

            if (sortConfig.key === 'lastModified') {
                 const dateA = new Date(a.last_modified || 0).getTime();
                 const dateB = new Date(b.last_modified || 0).getTime();
                 return sortConfig.direction === 'asc' ? dateA - dateB : dateB - dateA;
            }

            if (typeof aVal === 'string' && typeof bVal === 'string') {
                if (aVal.toLowerCase() < bVal.toLowerCase()) return sortConfig.direction === 'asc' ? -1 : 1;
                if (aVal.toLowerCase() > bVal.toLowerCase()) return sortConfig.direction === 'asc' ? 1 : -1;
            }
            
            return 0;
        });

    }, [sentEmails, users, searchTerm, sortConfig]);

    const filteredLogs = useMemo(() => {
        if (activeTab === 'all') return mappedAndSortedLogs;
        if (activeTab === 'campaigns') return mappedAndSortedLogs.filter(log => log.audience.source !== 'One-Off');
        if (activeTab === 'one-offs') return mappedAndSortedLogs.filter(log => log.audience.source === 'One-Off');
        return [];
    }, [mappedAndSortedLogs, activeTab]);

    const handleViewClick = (campaign: Campaign) => {
        setSelectedCampaign(campaign);
        setIsViewOpen(true);
    };

    const handleSort = (key: SortConfig['key'], direction: SortConfig['direction']) => {
      setSortConfig({ key, direction });
    }

    const sortLabels: Record<string, string> = {
        'lastModified-desc': 'Latest',
        'lastModified-asc': 'Oldest',
        'recipient-asc': 'Recipient (A-Z)',
        'recipient-desc': 'Recipient (Z-A)',
    }
    const currentSortLabel = sortLabels[`${sortConfig.key}-${sortConfig.direction}`] || 'Sort by';


    return (
        <div className="flex flex-col h-full">
            <PageHeader
                title="Sent Emails Log"
                description="A real-time log of all emails sent through the system."
            />
            <Card className="flex-1">
                 <CardHeader>
                    <div className="flex gap-2">
                        <div className="relative flex-1">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                            <Input
                                placeholder="Search by user or recipient..."
                                className="pl-10"
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                            />
                        </div>
                         <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                                <Button variant="outline">{currentSortLabel} <ChevronDown className="ml-2 h-4 w-4" /></Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                                <DropdownMenuItem onClick={() => handleSort('lastModified', 'desc')}>
                                <Clock className="mr-2 h-4 w-4" /> Latest
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('lastModified', 'asc')}>
                                <Clock className="mr-2 h-4 w-4" /> Oldest
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('recipient', 'asc')}>
                                <ArrowDownAZ className="mr-2 h-4 w-4" /> Recipient (A-Z)
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('recipient', 'desc')}>
                                <ArrowUpAZ className="mr-2 h-4 w-4" /> Recipient (Z-A)
                                </DropdownMenuItem>
                            </DropdownMenuContent>
                        </DropdownMenu>
                    </div>
                </CardHeader>
                <CardContent>
                    <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
                        <TabsList>
                            <TabsTrigger value="all">All ({mappedAndSortedLogs.length})</TabsTrigger>
                            <TabsTrigger value="campaigns">Campaigns ({mappedAndSortedLogs.filter(log => log.audience.source !== 'One-Off').length})</TabsTrigger>
                            <TabsTrigger value="one-offs">One-Offs ({mappedAndSortedLogs.filter(log => log.audience.source === 'One-Off').length})</TabsTrigger>
                        </TabsList>
                        <TabsContent value="all" className="mt-0">
                           <SentEmailsTable logs={filteredLogs} onActionClick={handleViewClick} />
                        </TabsContent>
                         <TabsContent value="campaigns" className="mt-0">
                           <SentEmailsTable logs={filteredLogs} onActionClick={handleViewClick} />
                        </TabsContent>
                         <TabsContent value="one-offs" className="mt-0">
                           <SentEmailsTable logs={filteredLogs} onActionClick={handleViewClick} />
                        </TabsContent>
                    </Tabs>
                </CardContent>
            </Card>
            <ViewSentEmailDialog
                campaign={selectedCampaign}
                open={isViewOpen}
                onOpenChange={setIsViewOpen}
            />
        </div>
    )
}
