'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import { getUserId } from '@/lib/auth';
import type { Prospect, Template } from '@/lib/types';

interface OneOffSendData {
  prospect: Prospect;
  templateId: number;
  senderId: number;
  trackingDomainId?: number; // <-- ADD THIS LINE
  warmup: boolean;
  throttling: boolean;
}

/**
 * Creates and schedules a one-off email campaign in a single, safe transaction.
 */
export async function sendOneOffEmail(data: OneOffSendData) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: "Authentication failed." };

  const { prospect, templateId, senderId, trackingDomainId, warmup, throttling } = data;
  const client = await pool.connect();

  try {
    await client.query('BEGIN');

    // 1. Fetch the selected template content from the database
    const templateRes = await client.query<Template>(
      'SELECT subject, body FROM templates WHERE id = $1 AND user_id = $2',
      [templateId, userId]
    );
    if (templateRes.rows.length === 0) {
      throw new Error("Template not found or permission denied.");
    }
    const template = templateRes.rows[0];

    // 2. Create a new, hidden "One-Off" audience list for this single prospect
    const audienceRes = await client.query(
      `INSERT INTO audiences (user_id, name, status, source, count, created_at, last_modified)
       VALUES ($1, $2, 'Saved', 'One-Off', 1, NOW(), NOW())
       RETURNING id`,
      [userId, `One-Off Audience for ${prospect.email}`]
    );
    const newAudienceId = audienceRes.rows[0].id;

    // 3. Add the single prospect to this new audience list
    await client.query(
      'INSERT INTO prospects (audience_id, name, email) VALUES ($1, $2, $3)',
      [newAudienceId, prospect.name, prospect.email]
    );

    // 4. Create a new campaign, mark it as "Scheduled" for the worker to pick up
    const campaignName = `One-Off: ${template.subject}`;
    const sequence = {
      steps: [{
        id: `step_${Date.now()}`,
        name: 'Initial Email',
        delay: 0,
        condition: 'not-opened',
        templateId: templateId,
        subject: template.subject,
        body: template.body
      }]
    };

    const campaignRes = await client.query(
      `INSERT INTO campaigns (user_id, name, objective, status, audience_list_id, sequence, warmup, throttling, tracking_domain_id)
       VALUES ($1, $2, 'Sales Conversion', 'Scheduled', $3, $4::jsonb, $5, $6, $7)
       RETURNING id`,
      [userId, campaignName, newAudienceId, JSON.stringify(sequence), warmup, throttling, trackingDomainId]
    );
    const newCampaignId = campaignRes.rows[0].id;

    // 5. Link the single chosen sender to the new campaign
    await client.query(
        'INSERT INTO campaign_senders (campaign_id, smtp_service_id) VALUES ($1, $2)',
        [newCampaignId, senderId]
    );

    await client.query('COMMIT');
    revalidatePath('/campaigns');
    return { success: true, message: 'Email has been scheduled and will be sent shortly.' };

  } catch (error) {
    await client.query('ROLLBACK');
    console.error("Database Error: Failed to send one-off email.", error);
    const errorMessage = error instanceof Error ? error.message : 'A database error occurred.';
    return { success: false, message: errorMessage };
  } finally {
    client.release();
  }
}