

'use client';

import Link from 'next/link';
import { usePathname, useRouter } from 'next/navigation';
import {
  SidebarProvider,
  Sidebar,
  SidebarHeader,
  SidebarTrigger,
  SidebarMenu,
  SidebarMenuItem,
  SidebarMenuButton,
  SidebarFooter,
  SidebarInset,
} from '@/components/ui/sidebar';
import {
  Avatar,
  AvatarFallback,
  AvatarImage,
} from '@/components/ui/avatar';
import {
  BarChart2,
  FileText,
  Mail,
  Settings,
  LogOut,
  Users,
  Wand2,
  LayoutTemplate,
  HeartPulse,
  Send,
  Shield,
  MessageSquare,
  KeyRound,
  Gem,
  X,
  UserCheck,
  Rocket,
  Mails,
  Globe
} from 'lucide-react';
import { Logo } from '@/components/logo';
import { Tooltip, TooltipContent, TooltipTrigger } from '@/components/ui/tooltip';
import { useEffect, useState } from 'react';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Separator } from '@/components/ui/separator';
import { CampaignProvider } from '@/context/CampaignContext';
import { AudienceProvider } from '@/context/AudienceContext';
import { UserProvider } from '@/context/UserContext';
import { ActivityLogProvider } from '@/context/ActivityLogContext';

import { TemplateProvider } from '@/context/TemplateContext';

const navItems = [
  { href: '/ai-drafts', label: 'AI Mail Generation', icon: Wand2 },
  { href: '/templates', label: 'Templates', icon: LayoutTemplate },
  { href: '/prospects', label: 'Prospects', icon: UserCheck },
  { href: '/audience', label: 'Audience', icon: Users },
  { href: '/campaigns', label: 'Campaigns', icon: Rocket },
  { href: '/one-off', label: 'One-off Email Sender', icon: Send },
  { href: '/sent-emails', label: 'Sent Emails Log', icon: Mails },
  { href: '/smart-reply-inbox', label: 'Smart Reply Inbox', icon: MessageSquare },
  { href: '/delivery-health', label: 'Delivery Health', icon: HeartPulse },
  { href: '/analytics', label: 'Analytics', icon: BarChart2 },
];

const adminNavItems = [
    { href: '/admin', label: 'Admin', icon: Shield },
];

export default function AppLayout({ children }: { children: React.ReactNode }) {
  const pathname = usePathname();
  const router = useRouter();
  const [userRole, setUserRole] = useState<string | null>(null);
  const [userName, setUserName] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState<string | null>(null);
  const [profilePic, setProfilePic] = useState<string | null>(null);
  const [isPremiumCardVisible, setIsPremiumCardVisible] = useState(true);

  const fetchUserData = () => {
    if (typeof window !== 'undefined') {
        const role = sessionStorage.getItem('userRole');
        const name = sessionStorage.getItem('userName');
        const email = sessionStorage.getItem('userEmail');
        const pic = sessionStorage.getItem('profilePic');
        setUserRole(role);
        setUserName(name);
        setUserEmail(email);
        setProfilePic(pic);
    }
  }
  
  useEffect(() => {
    if ('scrollRestoration' in history) {
      history.scrollRestoration = 'manual';
    }
    window.scrollTo(0, 0);
  }, [pathname]);

  useEffect(() => {
    fetchUserData();

    const handleStorageChange = () => {
      fetchUserData();
    };

    window.addEventListener('storage', handleStorageChange);
    // Add custom event listener for session storage changes
    window.addEventListener('sessionStorageChanged', handleStorageChange);

    return () => {
        window.removeEventListener('storage', handleStorageChange);
        window.removeEventListener('sessionStorageChanged', handleStorageChange);
    };
  }, []);

  const handleLogout = () => {
    if (typeof window !== 'undefined') {
        sessionStorage.clear();
    }
    router.push('/login');
  }

  const getInitials = (name?: string | null) => {
    if (!name) return 'U';
    return name
        .split(' ')
        .map(n => n[0])
        .slice(0, 2)
        .join('')
        .toUpperCase();
  }

  const homeHref = userRole === 'Admin' ? '/admin' : '/campaigns';

  return (
    <UserProvider>
      <CampaignProvider>
        <AudienceProvider>
            <ActivityLogProvider>
              <TemplateProvider>

                  <SidebarProvider>
                    <Sidebar side="left" variant="sidebar" collapsible="icon">
                      <SidebarHeader>
                        <Link href={homeHref} className="block" aria-label="Go to dashboard">
                          <Logo iconOnly className="group-data-[collapsible=icon]:block hidden" />
                          <Logo className="group-data-[collapsible=icon]:hidden" />
                        </Link>
                      </SidebarHeader>
                      <SidebarMenu className="flex-1 mt-4">
                        {navItems.map((item) => (
                          <SidebarMenuItem key={item.label}>
                            <Link href={item.href} passHref>
                              <SidebarMenuButton
                                asChild
                                isActive={pathname ? pathname.startsWith(item.href) && item.label !== 'Campaign Email Sender' : false}
                                tooltip={item.label}
                              >
                                <span>
                                  <item.icon />
                                  {item.label}
                                </span>
                              </SidebarMenuButton>
                            </Link>
                          </SidebarMenuItem>
                        ))}
                        {userRole === 'Admin' && adminNavItems.map((item) => (
                          <SidebarMenuItem key={item.href}>
                            <Link href={item.href} passHref>
                              <SidebarMenuButton
                                asChild
                                isActive={pathname ? pathname.startsWith(item.href) : false}
                                tooltip={item.label}
                              >
                                <span>
                                  <item.icon />
                                  {item.label}
                                </span>
                              </SidebarMenuButton>
                            </Link>
                          </SidebarMenuItem>
                        ))}
                      </SidebarMenu>

                      {userRole !== 'Admin' && isPremiumCardVisible && (
                        <div className="px-2 my-4 group-data-[collapsible=icon]:hidden">
                          <Card className="bg-amber-800/20 border-amber-600/50 text-sidebar-foreground relative">
                              <Button variant="ghost" size="icon" className="absolute top-1 right-1 h-6 w-6 text-sidebar-foreground/70 hover:text-sidebar-foreground hover:bg-transparent" onClick={() => setIsPremiumCardVisible(false)}>
                                  <X className="h-4 w-4" />
                              </Button>
                              <CardContent className="p-4">
                                <p className="text-2xl font-bold">10x</p>
                                <p className="text-sm mt-1">
                                    Premium members are 10x more likely to close a deal.
                                </p>
                                <Button variant="link" className="p-0 h-auto mt-2 text-sidebar-foreground/90 hover:text-sidebar-foreground">
                                      <Gem className="mr-2 h-4 w-4 text-yellow-400"/>
                                      Upgrade to Premium
                                </Button>
                              </CardContent>
                          </Card>
                        </div>
                      )}

                      <SidebarFooter>
                        <SidebarMenu>
                          <SidebarMenuItem>
                            <Link href="/settings" passHref>
                              <SidebarMenuButton
                                asChild
                                isActive={pathname ? pathname.startsWith('/settings') : false}
                                tooltip="Settings"
                              >
                                <span>
                                  <Settings />
                                  Settings
                                </span>
                              </SidebarMenuButton>
                            </Link>
                          </SidebarMenuItem>
                          <SidebarMenuItem>
                              <Link href="/settings/sending-settings" passHref>
                                  <SidebarMenuButton
                                      asChild
                                      isActive={pathname.startsWith('/settings/sending-settings')}
                                      tooltip="Sending Settings"
                                      >
                                      <span>
                                          <Send />
                                          Sending Settings
                                      </span>
                                  </SidebarMenuButton>
                              </Link>
                          </SidebarMenuItem>
                          <SidebarMenuItem>
                            <Link href="/settings/domains" passHref>
                                <SidebarMenuButton
                                    asChild
                                    isActive={pathname.startsWith('/settings/domains')}
                                    tooltip="Sending Domains"
                                    >
                                    <span>
                                        <Globe />
                                        Tracking Domains
                                    </span>
                                </SidebarMenuButton>
                            </Link>
                          </SidebarMenuItem>
                          <SidebarMenuItem>
                              <Link href="/settings/api-access" passHref>
                                  <SidebarMenuButton
                                      asChild
                                      isActive={pathname ? pathname.startsWith('/settings/api-access') : false}
                                      tooltip="API Access"
                                      >
                                      <span>
                                          <KeyRound />
                                          API Access
                                      </span>
                                  </SidebarMenuButton>
                              </Link>
                          </SidebarMenuItem>
                          <SidebarMenuItem>
                            <div className="flex items-center gap-3 p-2 group-data-[collapsible=icon]:p-0 group-data-[collapsible=icon]:justify-center">
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Avatar className="h-9 w-9">
                                    <AvatarImage src={profilePic || undefined} alt="User Avatar" />
                                    <AvatarFallback>{getInitials(userName)}</AvatarFallback>
                                  </Avatar>
                                </TooltipTrigger>
                                <TooltipContent side="right" align="center">
                                  <p>{userName || (userRole === 'Admin' ? 'Admin' : 'User')}</p>
                                  <p className="text-muted-foreground">{userEmail}</p>
                                </TooltipContent>
                              </Tooltip>
                            
                              <div className="flex flex-col group-data-[collapsible=icon]:hidden">
                                <span className="font-semibold text-sm text-sidebar-foreground">
                                  {userName || 'User'}
                                </span>
                                <span className="text-xs text-sidebar-foreground/70">
                                  {userEmail}
                                </span>
                              </div>
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <button onClick={handleLogout} className="ml-auto group-data-[collapsible=icon]:hidden">
                                      <LogOut className="w-5 h-5 text-sidebar-foreground/70 hover:text-sidebar-foreground" />
                                  </button>
                                </TooltipTrigger>
                                <TooltipContent side="right" align="center">
                                  Logout
                                </TooltipContent>
                              </Tooltip>
                            </div>
                          </SidebarMenuItem>
                        </SidebarMenu>
                      </SidebarFooter>
                    </Sidebar>
                    <SidebarInset>
                      <header className="flex h-14 items-center gap-4 border-b bg-background/95 backdrop-blur-sm sticky top-0 z-40 px-4 lg:px-6">
                        <SidebarTrigger className="md:hidden" />
                        <div className="flex-1">
                          {/* Can add breadcrumbs or page title here */}
                        </div>
                      </header>
                      <main className="flex-1 p-4 md:p-6 lg:p-8">{children}</main>
                    </SidebarInset>
                  </SidebarProvider>

              </TemplateProvider>
            </ActivityLogProvider>
        </AudienceProvider>
      </CampaignProvider>
    </UserProvider>
  );
}
