'use client';

import React, { useState, useMemo } from "react";
import { PlusCircle, MoreHorizontal, FileEdit, Send, Trash2, Settings, LayoutGrid, List, Search, ChevronDown, Clock, ArrowDownAZ, ArrowUpAZ, FileQuestion, Users2, Mails, Book, Server, Flame, Gauge } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { PageHeader } from "@/components/page-header";
import type { Campaign, CampaignStatus } from "@/lib/types";
import { Separator } from "@/components/ui/separator";
import { useCampaigns } from "@/context/CampaignContext";
import { cn } from "@/lib/utils";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { format } from 'date-fns';
import { Input } from "@/components/ui/input";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { CreateCampaignDialog } from "@/components/create-campaign-dialog";
import { EditCampaignDialog } from "@/components/edit-campaign-dialog";
import { useToast } from "@/hooks/use-toast";

type SortConfig = {
    key: keyof Campaign | 'lastModified';
    direction: 'asc' | 'desc';
} | null;

// All sub-components (CampaignTableRow, CampaignCard) remain the same.
const CampaignTableRow = ({ campaign, onActionSelect }: { campaign: Campaign, onActionSelect: (campaign: Campaign, action: 'edit' | 'delete' | 'send') => void }) => {
  const audienceName = campaign.audience?.name || 'No list selected';
  const lastModified = campaign.lastModified ? format(new Date(campaign.lastModified), 'PPP') : 'N/A';
  return (
    <TableRow>
      <TableCell className="font-medium">{campaign.name}</TableCell>
      <TableCell><Badge className={cn({"bg-green-100 text-green-800": campaign.status === 'Scheduled', "bg-blue-100 text-blue-800": campaign.status === 'Sent', "bg-gray-100 text-gray-800": campaign.status === 'Draft'})}>{campaign.status}</Badge></TableCell>
      <TableCell>{campaign.objective}</TableCell>
      <TableCell>{audienceName}</TableCell>
      <TableCell>{lastModified}</TableCell>
      <TableCell className="text-right">
        <DropdownMenu>
            <DropdownMenuTrigger asChild><Button variant="ghost" className="h-8 w-8 p-0"><MoreHorizontal className="h-4 w-4" /></Button></DropdownMenuTrigger>
            <DropdownMenuContent align="end">
                <DropdownMenuItem onSelect={() => onActionSelect(campaign, 'edit')}><FileEdit className="mr-2 h-4 w-4" /> View/Edit</DropdownMenuItem>
                {campaign.status === 'Scheduled' && (<DropdownMenuItem onSelect={() => onActionSelect(campaign, 'send')}><Send className="mr-2 h-4 w-4" /> Send Now</DropdownMenuItem>)}
                 <DropdownMenuItem onSelect={() => onActionSelect(campaign, 'delete')} className="text-destructive"><Trash2 className="mr-2 h-4 w-4" /> Delete</DropdownMenuItem>
            </DropdownMenuContent>
        </DropdownMenu>
      </TableCell>
    </TableRow>
  )
}
const CampaignCard = ({ campaign, onActionSelect }: { campaign: Campaign, onActionSelect: (campaign: Campaign, action: 'edit' | 'delete' | 'send') => void }) => {
    const audienceName = campaign.audience?.name || 'No list selected';
    const lastModified = campaign.lastModified ? format(new Date(campaign.lastModified), 'PPP') : 'N/A';
    const templatesUsedCount = campaign.sequence?.steps?.filter(step => step.templateId).length ?? 0;
    return (
        <Card>
            <CardHeader><div className="flex justify-between items-start"><div><CardTitle>{campaign.name}</CardTitle><CardDescription>Last modified on {lastModified}</CardDescription></div><Badge className={cn({"bg-green-100 text-green-800": campaign.status === 'Scheduled', "bg-blue-100 text-blue-800": campaign.status === 'Sent', "bg-gray-100 text-gray-800": campaign.status === 'Draft'})}>{campaign.status}</Badge></div></CardHeader>
            <CardContent className="grid gap-4 text-sm text-muted-foreground"><div className="flex items-center gap-2"><FileQuestion className="h-4 w-4" /><span>Objective: <strong>{campaign.objective}</strong></span></div><div className="flex items-center gap-2"><Users2 className="h-4 w-4" /><span>Audience: <strong>{audienceName}</strong></span></div><Separator /><div className="grid grid-cols-2 gap-4"><div className="flex items-center gap-2"><Mails className="h-4 w-4" /><span>{campaign.sequence?.steps?.length ?? 0} Steps</span></div><div className="flex items-center gap-2"><Book className="h-4 w-4" /><span>{templatesUsedCount} Templates</span></div><div className="flex items-center gap-2"><Flame className="h-4 w-4" /><span>Warmup: {campaign.sender?.warmup ? 'On' : 'Off'}</span></div><div className="flex items-center gap-2"><Gauge className="h-4 w-4" /><span>Throttling: {campaign.sender?.throttling ? 'On' : 'Off'}</span></div></div></CardContent>
            <Separator className="my-4" /><CardFooter className="justify-end space-x-2"><DropdownMenu><DropdownMenuTrigger asChild><Button variant="outline" size="sm"><Settings className="mr-2 h-4 w-4" /> Actions</Button></DropdownMenuTrigger><DropdownMenuContent align="end"><DropdownMenuItem onSelect={() => onActionSelect(campaign, 'edit')}><FileEdit className="mr-2 h-4 w-4" /> View/Edit</DropdownMenuItem>{campaign.status === 'Scheduled' && (<DropdownMenuItem onSelect={() => onActionSelect(campaign, 'send')}><Send className="mr-2 h-4 w-4" /> Send Now</DropdownMenuItem>)}<DropdownMenuItem onSelect={() => onActionSelect(campaign, 'delete')} className="text-destructive"><Trash2 className="mr-2 h-4 w-4" /> Delete</DropdownMenuItem></DropdownMenuContent></DropdownMenu></CardFooter>
        </Card>
    );
};

// --- THIS COMPONENT IS NOW FIXED ---
const CampaignList = ({ campaigns, status, viewMode, onActionSelect }: { campaigns: Campaign[], status?: CampaignStatus, viewMode: 'card' | 'list', onActionSelect: (campaign: Campaign, action: 'edit' | 'delete' | 'send') => void }) => {
    // The filter logic is now explicit for each status. "all" shows everything.
    const filteredCampaigns = status ? campaigns.filter(c => c.status === status) : campaigns;

    if (viewMode === 'list') {
        return (
            <Card>
                <Table>
                    <TableHeader><TableRow><TableHead>Campaign Name</TableHead><TableHead>Status</TableHead><TableHead>Objective</TableHead><TableHead>Audience</TableHead><TableHead>Last Modified</TableHead><TableHead className="text-right">Actions</TableHead></TableRow></TableHeader>
                    <TableBody>
                        {filteredCampaigns.length > 0 ? (
                            filteredCampaigns.map((campaign) => (<CampaignTableRow key={campaign.id} campaign={campaign} onActionSelect={onActionSelect} />))
                        ) : ( <TableRow><TableCell colSpan={6} className="h-24 text-center">No campaigns found for this status.</TableCell></TableRow> )}
                    </TableBody>
                </Table>
            </Card>
        )
    }
    if (filteredCampaigns.length === 0) return (<Card><CardContent className="p-6 text-center text-muted-foreground">No campaigns found for this status.</CardContent></Card>)
    return (<div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">{filteredCampaigns.map((campaign) => (<CampaignCard key={campaign.id} campaign={campaign} onActionSelect={onActionSelect} />))}</div>);
};


export default function CampaignsPage() {
  const { campaigns, isLoading, fetchCampaigns } = useCampaigns();
  const [viewMode, setViewMode] = useState<'card' | 'list'>('list');
  const [searchTerm, setSearchTerm] = useState('');
  const [sortConfig, setSortConfig] = useState<SortConfig>({ key: 'lastModified', direction: 'desc' });
  const [isCreateOpen, setIsCreateOpen] = useState(false);
  const [campaignToEdit, setCampaignToEdit] = useState<Campaign | null>(null);
  const [isEditOpen, setIsEditOpen] = useState(false);
  const [campaignToDelete, setCampaignToDelete] = useState<Campaign | null>(null);
  const [isDeleteAlertOpen, setIsDeleteAlertOpen] = useState(false);
  const { toast } = useToast();

  const handleAction = (campaign: Campaign, action: 'edit' | 'delete' | 'send') => {
      if (action === 'edit') { setCampaignToEdit(campaign); setIsEditOpen(true); }
      else if (action === 'delete') { setCampaignToDelete(campaign); setIsDeleteAlertOpen(true); }
  };

  const handleSaveEdit = async (editedCampaign: Partial<Campaign>, newStatus?: CampaignStatus) => {
      console.log("Saving edited campaign:", editedCampaign, "New status:", newStatus);
      await fetchCampaigns();
      toast({ title: 'Campaign Updated' });
      setIsEditOpen(false);
  };

  const handleDeleteConfirm = async () => {
    if (campaignToDelete) {
        console.log("Deleting campaign:", campaignToDelete.id);
        await fetchCampaigns();
        toast({ variant: 'destructive', title: 'Campaign Deleted' });
        setIsDeleteAlertOpen(false);
        setCampaignToDelete(null);
    }
  };

  const sortedCampaigns = useMemo(() => {
    return campaigns
        .filter(c => c.name.toLowerCase().includes(searchTerm.toLowerCase()))
        .sort((a, b) => {
            if (!sortConfig) return 0;
            if (sortConfig.key === 'lastModified') {
                return sortConfig.direction === 'asc'
                    ? new Date(a.lastModified).getTime() - new Date(b.lastModified).getTime()
                    : new Date(b.lastModified).getTime() - new Date(a.lastModified).getTime();
            }
            const aValue = a[sortConfig.key as keyof Campaign];
            const bValue = b[sortConfig.key as keyof Campaign];
            if (typeof aValue === 'string' && typeof bValue === 'string') {
                return sortConfig.direction === 'asc' ? aValue.localeCompare(bValue) : bValue.localeCompare(aValue);
            }
            return 0;
        });
  }, [campaigns, sortConfig, searchTerm]);

  const scheduledCampaigns = sortedCampaigns.filter(c => c.status === 'Scheduled');
  const draftCampaigns = sortedCampaigns.filter(c => c.status === 'Draft');
  const sentCampaigns = sortedCampaigns.filter(c => c.status === 'Sent'); // <-- New filtered list

  if (isLoading) {
      return <PageHeader title="Campaigns" description="Loading your campaigns..." />
  }

  return (
    <>
      <PageHeader title="Campaigns" description="Manage and track your email campaigns.">
        <div className="flex items-center gap-2">
            <Button onClick={() => setIsCreateOpen(true)}><PlusCircle className="mr-2 h-4 w-4" />Create Campaign</Button>
        </div>
      </PageHeader>

       <Tabs defaultValue="all" className="space-y-4">
        {/* --- THIS IS THE KEY UI FIX --- */}
        <TabsList>
            <TabsTrigger value="all">All Campaigns ({sortedCampaigns.length})</TabsTrigger>
            <TabsTrigger value="scheduled">Scheduled ({scheduledCampaigns.length})</TabsTrigger>
            <TabsTrigger value="drafts">Drafts ({draftCampaigns.length})</TabsTrigger>
            <TabsTrigger value="sent">Sent ({sentCampaigns.length})</TabsTrigger> {/* <-- ADDED THIS TAB */}
        </TabsList>
        <TabsContent value="all" className="mt-0">
          <CampaignList campaigns={sortedCampaigns} viewMode={viewMode} onActionSelect={handleAction} />
        </TabsContent>
         <TabsContent value="scheduled" className="mt-0">
          <CampaignList campaigns={sortedCampaigns} status="Scheduled" viewMode={viewMode} onActionSelect={handleAction} />
        </TabsContent>
        <TabsContent value="drafts" className="mt-0">
           <CampaignList campaigns={sortedCampaigns} status="Draft" viewMode={viewMode} onActionSelect={handleAction} />
        </TabsContent>
        <TabsContent value="sent" className="mt-0">
           <CampaignList campaigns={sortedCampaigns} status="Sent" viewMode={viewMode} onActionSelect={handleAction} /> {/* <-- ADDED THIS CONTENT */}
        </TabsContent>
      </Tabs>
      <CreateCampaignDialog open={isCreateOpen} onOpenChange={setIsCreateOpen} />
      <EditCampaignDialog campaign={campaignToEdit} open={isEditOpen} onOpenChange={setIsEditOpen} onSave={handleSaveEdit} />
      <AlertDialog open={isDeleteAlertOpen} onOpenChange={setIsDeleteAlertOpen}>
          <AlertDialogContent>
              <AlertDialogHeader><AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle><AlertDialogDescription>This will permanently delete the campaign "{campaignToDelete?.name}".</AlertDialogDescription></AlertDialogHeader>
              <AlertDialogFooter><AlertDialogCancel onClick={() => setCampaignToDelete(null)}>Cancel</AlertDialogCancel><AlertDialogAction onClick={handleDeleteConfirm}>Delete</AlertDialogAction></AlertDialogFooter>
          </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
