'use client';

import { useState, useMemo, useEffect } from "react";
import Link from "next/link";
import {
  PlusCircle, FileEdit, Download, Database, Eye, LayoutGrid, List, Search,
  ChevronDown, Clock, ArrowDownAZ, ArrowUpAZ, Users
} from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter
} from "@/components/ui/card";
import {
  Table, TableBody, TableCell, TableHead, TableHeader, TableRow
} from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { PageHeader } from "@/components/page-header";
import type { AudienceList, AudienceListStatus, Prospect } from "@/lib/types";
import { Separator } from "@/components/ui/separator";
import { cn } from "@/lib/utils";
import { useToast } from "@/hooks/use-toast";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import {
  DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuLabel,
  DropdownMenuSeparator, DropdownMenuTrigger
} from "@/components/ui/dropdown-menu";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { ScrollArea } from "@/components/ui/scroll-area";
import { format } from 'date-fns';

// --- SERVER ACTION ---
// CORRECTED PATH: Use './actions' to look in the current directory
import { getAudienceLists } from './actions';

// --- This is your excellent Prospect Viewer Dialog ---
function ProspectViewerDialog({
  open,
  onOpenChange,
  prospects,
  listName,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  prospects: Prospect[];
  listName: string;
}) {
  const [searchTerm, setSearchTerm] = useState('');

  const filteredProspects = useMemo(() => {
    if (!searchTerm) return prospects;
    return prospects.filter(p =>
      p.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      p.email.toLowerCase().includes(searchTerm.toLowerCase())
    );
  }, [prospects, searchTerm]);

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-2xl">
        <DialogHeader>
          <DialogTitle>Prospects in "{listName}"</DialogTitle>
          <DialogDescription>
            Showing {filteredProspects.length} of {prospects.length} total prospects.
          </DialogDescription>
        </DialogHeader>
        <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
                placeholder="Search prospects..."
                className="pl-10"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
            />
        </div>
        <ScrollArea className="h-96 pr-4">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Name</TableHead>
                <TableHead>Email</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredProspects.length > 0 ? (
                filteredProspects.map(p => (
                  // Use a more robust key, like email if ID isn't guaranteed unique from old data
                  <TableRow key={p.id || p.email}>
                    <TableCell className="font-medium">{p.name}</TableCell>
                    <TableCell>{p.email}</TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={2} className="h-24 text-center">
                    No prospects found.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </ScrollArea>
      </DialogContent>
    </Dialog>
  );
}


const AudienceListCard = ({ list }: { list: AudienceList }) => {
  const isDraft = list.status === 'Draft';
  const [isProspectsOpen, setIsProspectsOpen] = useState(false);
  // The 'prospects' array is now guaranteed to be complete thanks to our server action
  const prospects = list.prospects || [];

  return (
    <>
      <Card>
        <CardHeader>
          <div className="flex justify-between items-start">
              <div>
                <div className="flex items-center gap-2"><CardTitle>{list.name}</CardTitle>{list.source === 'EP DB' && (<Badge variant="outline" className="flex items-center gap-1 text-primary border-primary/50"><Database className="h-3 w-3" />EP DB</Badge>)}</div>
                <CardDescription>Last modified on {format(new Date(list.last_modified), 'PPP')}</CardDescription>
              </div>
              <Badge className={cn({ "bg-blue-100 text-blue-800": list.status === 'Saved', "bg-gray-100 text-gray-800": list.status === 'Draft' })}>{list.status}</Badge>
          </div>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
              <p className="text-sm font-medium">Criteria</p>
              <div className="flex flex-wrap gap-1">{list.criteria && list.criteria.length > 0 ? list.criteria.map(c => (<Badge key={c.id} variant="secondary">{c.value}</Badge>)) : <p className="text-sm text-muted-foreground">No criteria.</p>}</div>
          </div>
        </CardContent>
          <Separator className="my-4" />
          <CardFooter className="justify-between items-center">
              <p className="text-sm text-muted-foreground">Audience Count: <span className="font-semibold text-foreground">~{list.count.toLocaleString()}</span></p>
              <div className="flex items-center gap-2">
                  {/* This button now works for ALL audiences */}
                  <Button variant="secondary" onClick={() => setIsProspectsOpen(true)} disabled={prospects.length === 0}>
                      <Users className="mr-2 h-4 w-4" />
                      View Prospects
                  </Button>
                  <Link href={`/audience/edit/${list.id}`}><Button variant="outline"><Eye className="mr-2 h-4 w-4" />{isDraft ? 'Edit' : 'View'}</Button></Link>
              </div>
          </CardFooter>
      </Card>
      <ProspectViewerDialog
        open={isProspectsOpen}
        onOpenChange={setIsProspectsOpen}
        prospects={prospects}
        listName={list.name}
      />
    </>
  )
}

const AudienceListGrid = ({ lists, status, viewMode }: { lists: AudienceList[], status?: AudienceListStatus, viewMode: 'card' | 'list' }) => {
    // This component remains the same
    const filteredLists = status ? lists.filter(c => c.status === status) : lists;
    if (filteredLists.length === 0) { return (<Card><CardContent className="p-6 text-center text-muted-foreground">No audience lists found.</CardContent></Card>) }
    if (viewMode === 'list') {
        // Implement Table view if needed
        return <p>List view not implemented yet.</p>;
    }
    return <div className="space-y-4">{filteredLists.map(list => <AudienceListCard key={list.id} list={list} />)}</div>;
};

export default function AudiencePage() {
    // --- THIS IS THE MAIN FIX ---
    // We now manage state fetched directly from the server action
    const [audienceLists, setAudienceLists] = useState<AudienceList[]>([]);
    const [isLoading, setIsLoading] = useState(true);
    const { toast } = useToast();
    const [viewMode, setViewMode] = useState<'card' | 'list'>('card');
    const [searchTerm, setSearchTerm] = useState('');

    useEffect(() => {
        setIsLoading(true);
        // Fetch data when the component mounts
        getAudienceLists()
            .then(data => {
                setAudienceLists(data);
            })
            .catch(() => {
                toast({ variant: 'destructive', title: 'Error', description: 'Could not fetch audience lists from the database.' });
            })
            .finally(() => {
                setIsLoading(false);
            });
    }, [toast]);

    const sortedLists = useMemo(() => {
        return audienceLists
            .filter(list => list.name.toLowerCase().includes(searchTerm.toLowerCase()))
            .sort((a, b) => new Date(b.last_modified).getTime() - new Date(a.last_modified).getTime());
    }, [audienceLists, searchTerm]);

    const savedLists = sortedLists.filter(l => l.status === 'Saved');
    const draftLists = sortedLists.filter(l => l.status === 'Draft');

    if (isLoading) {
        return <PageHeader title="Audience Lists" description="Loading your audience lists from the database..." />;
    }

    return (
        <>
            <PageHeader title="Audience Lists" description="Manage and analyze your target audiences.">
                 <div className="flex items-center gap-2">
                    {/* Add other header buttons like sort/filter if needed */}
                    <Link href="/audience/new"><Button><PlusCircle className="mr-2 h-4 w-4" />New Audience List</Button></Link>
                 </div>
            </PageHeader>
            <div className="relative w-full mb-4">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input placeholder="Search audience lists by name..." className="pl-10" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
            </div>
            <Tabs defaultValue="all">
                <TabsList className="mb-4">
                    <TabsTrigger value="all">All Lists ({sortedLists.length})</TabsTrigger>
                    <TabsTrigger value="saved">Saved ({savedLists.length})</TabsTrigger>
                    <TabsTrigger value="drafts">Drafts ({draftLists.length})</TabsTrigger>
                </TabsList>
                <TabsContent value="all"><AudienceListGrid lists={sortedLists} viewMode={viewMode}/></TabsContent>
                <TabsContent value="saved"><AudienceListGrid lists={sortedLists} status="Saved" viewMode={viewMode}/></TabsContent>
                <TabsContent value="drafts"><AudienceListGrid lists={sortedLists} status="Draft" viewMode={viewMode}/></TabsContent>
            </Tabs>
        </>
    );
}

