'use server';

import { pool } from '@/lib/db';
import { unstable_noStore as noStore } from 'next/cache';

export async function getAnalyticsData() {
  // Prevent this data from being cached to ensure it's always fresh.
  noStore();

  try {
    // 1. Fetch Overall Performance for the last 7 days (for the line chart)
    const performanceQuery = `
      SELECT
        TO_CHAR(timestamp, 'YYYY-MM-DD') AS date,
        COUNT(*) FILTER (WHERE event_type = 'SENT') AS sent,
        COUNT(*) FILTER (WHERE event_type = 'OPEN') AS opens,
        COUNT(*) FILTER (WHERE event_type = 'CLICK') AS clicks
      FROM email_events
      WHERE timestamp >= NOW() - INTERVAL '7 days'
      GROUP BY date
      ORDER BY date ASC;
    `;
    const performanceRes = await pool.query(performanceQuery);

    // 2. Fetch Key Metric totals (for the cards)
    const totalsQuery = `
      SELECT
        COUNT(*) FILTER (WHERE event_type = 'REPLY') AS "totalReplies",
        COUNT(*) FILTER (WHERE event_type = 'BOUNCE') AS "totalBounces",
        COUNT(*) FILTER (WHERE event_type = 'UNSUBSCRIBE') AS "totalUnsubscribes"
      FROM email_events;
    `;
    const totalsRes = await pool.query(totalsQuery);

    // 3. Fetch Performance by Campaign (for the table)
    const byCampaignQuery = `
      SELECT
        c.id,
        c.name,
        c.status,
        COUNT(e.id) FILTER (WHERE e.event_type = 'SENT') AS sent,
        -- Calculate open rate: (opens / sent) * 100
        COALESCE(
            (COUNT(e.id) FILTER (WHERE e.event_type = 'OPEN') * 100.0) / NULLIF(COUNT(e.id) FILTER (WHERE e.event_type = 'SENT'), 0),
            0
        ) AS "openRate",
        -- Calculate click rate: (clicks / sent) * 100
        COALESCE(
            (COUNT(e.id) FILTER (WHERE e.event_type = 'CLICK') * 100.0) / NULLIF(COUNT(e.id) FILTER (WHERE e.event_type = 'SENT'), 0),
            0
        ) AS "clickRate"
      FROM campaigns c
      LEFT JOIN email_events e ON c.id = e.campaign_id
      WHERE c.status IN ('Sent', 'Archived') -- Only show completed campaigns
      GROUP BY c.id, c.name, c.status
      ORDER BY c.last_modified DESC;
    `;
    const byCampaignRes = await pool.query(byCampaignQuery);

    // Note: One-off email performance would require a separate table or a way to identify them.
    // We will leave this as mock for now.

    return {
      campaignPerformance: performanceRes.rows,
      totals: totalsRes.rows[0],
      byCampaign: byCampaignRes.rows.map(row => ({
          ...row,
          // Format rates to 2 decimal places
          openRate: parseFloat(row.openRate).toFixed(2),
          clickRate: parseFloat(row.clickRate).toFixed(2),
      })),
    };

  } catch (error) {
    console.error("Database Error: Failed to fetch analytics data.", error);
    // Return a default structure on error so the page doesn't crash
    return {
      campaignPerformance: [],
      totals: { totalReplies: 0, totalBounces: 0, totalUnsubscribes: 0 },
      byCampaign: [],
    };
  }
}
