
'use client';

import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Separator } from "@/components/ui/separator";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useUsers } from "@/context/UserContext";
import { Copy, DollarSign, KeyRound, PlusCircle, Search, Trash2, X, ArrowLeft } from "lucide-react";
import { useState, useEffect } from "react";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import type { User, UserRole, UserStatus } from "@/lib/types";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import Link from "next/link";
import { cn } from "@/lib/utils";
import { useActivityLog } from "@/context/ActivityLogContext";

function ApiUsageDetailsDialog({ user, open, onOpenChange, onGenerateKey, onDeleteKey }: { user: User | null, open: boolean, onOpenChange: (open: boolean) => void, onGenerateKey: (userId: string) => void, onDeleteKey: (userId: string) => void }) {
    const { toast } = useToast();

    if (!user) return null;
    
    const handleCopyKey = (key: string) => {
        navigator.clipboard.writeText(key);
        toast({ title: "Copied!", description: "API Key has been copied to your clipboard." });
    }

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl">
                <DialogHeader>
                    <DialogTitle>API Details for {user.name}</DialogTitle>
                    <DialogDescription>
                        Manage API key, view usage, and payment details for this user.
                    </DialogDescription>
                </DialogHeader>
                <div className="grid gap-6 py-4">
                    <Card>
                        <CardHeader className="flex-row items-center justify-between pb-2">
                           <CardTitle className="text-base font-semibold">API Key</CardTitle>
                           <div className="flex gap-2">
                                <Button size="sm" variant="outline" onClick={() => onGenerateKey(user.id)}>
                                    <PlusCircle className="mr-2 h-4 w-4" /> Generate New
                                </Button>
                                {user.apiKey && (
                                     <Button size="sm" variant="destructive" onClick={() => onDeleteKey(user.id)}>
                                        <Trash2 className="mr-2 h-4 w-4" /> Delete Key
                                    </Button>
                                )}
                           </div>
                        </CardHeader>
                        <CardContent>
                             {user.apiKey ? (
                                <div className="flex items-center justify-between p-3 border rounded-lg bg-muted/50">
                                    <div className="flex items-center gap-3">
                                        <KeyRound className="h-5 w-5 text-muted-foreground" />
                                        <p className="font-mono text-sm text-muted-foreground">{user.apiKey}</p>
                                    </div>
                                    <Button variant="ghost" size="icon" onClick={() => handleCopyKey(user.apiKey)}><Copy className="h-4 w-4" /></Button>
                                </div>
                           ) : (
                                <p className="text-sm text-muted-foreground text-center py-4">No API key generated for this user.</p>
                           )}
                        </CardContent>
                    </Card>
                    <div className="grid grid-cols-2 gap-6">
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-base font-semibold">API Usage</CardTitle>
                                <CardDescription>Current billing cycle.</CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4 text-sm">
                                <div className="flex justify-between">
                                    <span>API Calls</span>
                                    <span className="font-mono">1,234 / 10,000</span>
                                </div>
                                <Separator />
                                <div className="flex justify-between">
                                    <span>Data Enrichment</span>
                                    <span className="font-mono">56 / 1,000</span>
                                </div>
                            </CardContent>
                        </Card>
                        <Card>
                             <CardHeader>
                                <CardTitle className="text-base font-semibold">Billing</CardTitle>
                                <CardDescription>Net total for this user.</CardDescription>
                            </CardHeader>
                            <CardContent className="flex items-center gap-2">
                                <DollarSign className="h-6 w-6 text-muted-foreground"/>
                                <p className="text-2xl font-bold">$199.99</p>
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </DialogContent>
        </Dialog>
    )
}

function UserRow({ user, onRowClick }: { user: User; onRowClick: (user: User) => void }) {
    const [status, setStatus] = useState<UserStatus | null>(null);

    useEffect(() => {
        if (user.role === 'Admin') {
            setStatus('Active');
        } else {
            setStatus(user.apiKey ? 'Active' : 'Inactive');
        }
    }, [user.apiKey, user.role]);

    const roleColors: { [key in UserRole]: string } = {
        Admin: 'bg-red-500 text-white',
        Manager: 'bg-purple-500 text-white',
        Staff: 'bg-blue-500 text-white',
        Consumer: 'bg-gray-200 text-gray-800',
    };

    if (!status) {
        return (
            <TableRow>
                <TableCell colSpan={4} className="h-12" />
            </TableRow>
        );
    }

    return (
        <TableRow onClick={() => onRowClick(user)} className="cursor-pointer">
            <TableCell className="font-medium">{user.name}</TableCell>
            <TableCell><Badge className={roleColors[user.role]}>{user.role}</Badge></TableCell>
            <TableCell className="font-mono text-muted-foreground">
                {user.apiKey ? `${user.apiKey.substring(0, 15)}...` : 'No key generated'}
            </TableCell>
            <TableCell>
                <Badge variant={status === 'Active' ? 'default' : 'secondary'} className={cn(
                    status === 'Active' && 'bg-green-600 text-white',
                    status === 'Inactive' && 'bg-gray-500 text-white hover:bg-primary/90'
                )}>{status}</Badge>
            </TableCell>
        </TableRow>
    );
}


export default function ApiUsagePage() {
    const { users, updateUser, fetchUsers } = useUsers();
    const { addLog, fetchLogs } = useActivityLog();
    const { toast } = useToast();
    const [searchTerm, setSearchTerm] = useState('');
    const [selectedUser, setSelectedUser] = useState<User | null>(null);
    const [isDialogOpen, setIsDialogOpen] = useState(false);

    const handleGenerateKey = async (userId: string) => {
        const user = users.find(u => u.id === userId);
        if (user) {
            const newApiKey = `sk_live_${crypto.randomUUID().replace(/-/g, '').substring(0, 24)}`;
            const updatedUserData = { apiKey: newApiKey };
            await updateUser(userId, updatedUserData);
            await fetchUsers();
            
            const updatedUser = users.find(u => u.id === userId);
            setSelectedUser(updatedUser || null);
            
            const adminUser = sessionStorage.getItem('userName') || 'Admin';
            await addLog({
                action: 'API Key Generated',
                userId: user.id,
                userName: adminUser,
                details: `A new API key was generated for ${user.name} by ${adminUser}.`
            });
            await fetchLogs();

            toast({ title: "API Key Generated", description: `A new key has been generated for ${user.name}.` });
        }
    };
    
    const handleDeleteKey = async (userId: string) => {
        const user = users.find(u => u.id === userId);
        if (user) {
            const updatedUserData = { apiKey: '' };
            await updateUser(userId, updatedUserData);
            await fetchUsers();
            
            const updatedUser = users.find(u => u.id === userId);
            setSelectedUser(updatedUser || null);

            const adminUser = sessionStorage.getItem('userName') || 'Admin';
            await addLog({
                action: 'API Key Deleted',
                userId: user.id,
                userName: adminUser,
                details: `API key was deleted for ${user.name} by ${adminUser}.`
            });
            await fetchLogs();

            toast({ variant: 'destructive', title: "API Key Deleted", description: `The API key for ${user.name} has been revoked.` });
        }
    };
    
    const handleRowClick = (user: User) => {
        setSelectedUser(user);
        setIsDialogOpen(true);
    };
    
    const filteredUsers = users.filter(user =>
        user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        user.email.toLowerCase().includes(searchTerm.toLowerCase())
    );

    return (
        <div className="flex flex-col h-full">
            <PageHeader
                title="API Usage Logs"
                description="Monitor and manage API keys for all users."
            />
            <Card className="flex-1">
                <CardHeader>
                    <div className="relative flex-1">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                        <Input
                            placeholder="Search users by name or email..."
                            className="pl-10 pr-10"
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                        />
                        {searchTerm && (
                            <Button
                                variant="ghost"
                                size="icon"
                                className="absolute right-2 top-1/2 -translate-y-1/2 h-6 w-6"
                                onClick={() => setSearchTerm('')}
                            >
                                <X className="h-4 w-4" />
                            </Button>
                        )}
                    </div>
                </CardHeader>
                <CardContent>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>User</TableHead>
                                <TableHead>Role</TableHead>
                                <TableHead>API Key</TableHead>
                                <TableHead>Status</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {filteredUsers.map(user => (
                                <UserRow key={user.id} user={user} onRowClick={handleRowClick} />
                            ))}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
            <div className="flex justify-end mt-6">
                <Link href="/admin">
                    <Button variant="outline">
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Admin Panel
                    </Button>
                </Link>
            </div>
            <ApiUsageDetailsDialog 
                user={selectedUser}
                open={isDialogOpen}
                onOpenChange={setIsDialogOpen}
                onGenerateKey={handleGenerateKey}
                onDeleteKey={handleDeleteKey}
            />
        </div>
    )
}
