
'use client';

import { useState, useEffect } from 'react';
import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Search, ArrowLeft, ArrowUpDown, Loader2 } from "lucide-react";
import Link from "next/link";
import { useActivityLog } from '@/context/ActivityLogContext';
import type { ActivityLog } from '@/lib/types';
import { format } from 'date-fns';

export default function ActivityLogsPage() {
    const { logs, isLoading, fetchLogs } = useActivityLog();
    const [searchTerm, setSearchTerm] = useState('');
    const [sortConfig, setSortConfig] = useState<{ key: keyof ActivityLog | 'timestamp'; direction: 'asc' | 'desc' } | null>({ key: 'timestamp', direction: 'desc' });

    useEffect(() => {
        fetchLogs();
    }, [fetchLogs]);

    const sortedLogs = [...logs].sort((a, b) => {
        if (!sortConfig) return 0;

        const aValue = a[sortConfig.key as keyof ActivityLog];
        const bValue = b[sortConfig.key as keyof ActivityLog];
        
        if (aValue < bValue) {
            return sortConfig.direction === 'asc' ? -1 : 1;
        }
        if (aValue > bValue) {
            return sortConfig.direction === 'asc' ? 1 : -1;
        }
        return 0;
    });
    
    const filteredLogs = sortedLogs.filter(log =>
        log.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        log.action.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (typeof log.details === 'string' ? log.details.toLowerCase().includes(searchTerm.toLowerCase()) : JSON.stringify(log.details).toLowerCase().includes(searchTerm.toLowerCase()))
    );

    const requestSort = (key: keyof ActivityLog | 'timestamp') => {
        let direction: 'asc' | 'desc' = 'asc';
        if (sortConfig && sortConfig.key === key && sortConfig.direction === 'asc') {
            direction = 'desc';
        }
        setSortConfig({ key, direction });
    };

    const getSortIndicator = (key: keyof ActivityLog | 'timestamp') => {
        if (!sortConfig || sortConfig.key !== key) {
            return <ArrowUpDown className="ml-2 h-4 w-4 opacity-30" />;
        }
        return sortConfig.direction === 'desc' ? '▼' : '▲';
    };


    return (
        <div className="flex flex-col h-full">
            <PageHeader
                title="Activity Logs"
                description="A permanent record of all account creations and modifications."
            />
            <Card className="flex-1">
                <CardHeader>
                    <div className="relative flex-1">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                        <Input
                            placeholder="Search logs by user, action, or details..."
                            className="pl-10"
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                        />
                    </div>
                </CardHeader>
                <CardContent>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>
                                    <Button variant="ghost" onClick={() => requestSort('timestamp')}>
                                        Timestamp {getSortIndicator('timestamp')}
                                    </Button>
                                </TableHead>
                                <TableHead>
                                    <Button variant="ghost" onClick={() => requestSort('userName')}>
                                        User {getSortIndicator('userName')}
                                    </Button>
                                </TableHead>
                                <TableHead>
                                     <Button variant="ghost" onClick={() => requestSort('action')}>
                                        Action {getSortIndicator('action')}
                                    </Button>
                                </TableHead>
                                <TableHead>Details</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {isLoading ? (
                                <TableRow>
                                    <TableCell colSpan={4} className="h-24 text-center">
                                        <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                                    </TableCell>
                                </TableRow>
                            ) : filteredLogs.length > 0 ? (
                                filteredLogs.map(log => (
                                    <TableRow key={log.id}>
                                        <TableCell className="font-mono text-xs">{format(new Date(log.timestamp), 'yyyy-MM-dd HH:mm:ss')}</TableCell>
                                        <TableCell className="font-medium">{log.userName} ({log.userId})</TableCell>
                                        <TableCell>{log.action}</TableCell>
                                        <TableCell className="text-sm text-muted-foreground">
                                            {typeof log.details === 'string' ? log.details : JSON.stringify(log.details)}
                                        </TableCell>
                                    </TableRow>
                                ))
                            ) : (
                                <TableRow>
                                    <TableCell colSpan={4} className="h-24 text-center">
                                        No activity logs found.
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
            <div className="flex justify-end mt-6">
                <Link href="/admin">
                    <Button variant="outline">
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Admin Panel
                    </Button>
                </Link>
            </div>
        </div>
    )
}
