'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import type { ActivityLog } from '@/lib/types'; // Make sure this type is defined in your types.ts
import { getUserId } from '@/lib/auth';

/**
 * --- THIS IS THE MISSING FUNCTION ---
 * Fetches activity logs from the database, joining with the users table
 * to get the name of the user who performed the action for a more informative log display.
 */
async function getActivityLogs(): Promise<ActivityLog[]> {
  try {
    const { rows } = await pool.query(`
      SELECT
        al.id,
        al.action,
        al.timestamp,
        al.details,
        u.name as user_name,
        u.email as user_email
      FROM activity_logs al
      JOIN users u ON al.user_id = u.id
      ORDER BY al.timestamp DESC
      LIMIT 100; -- Add a limit for performance on large logs
    `);

    // Map the database result (snake_case) to the expected UI type (camelCase)
    return rows.map(row => ({
        id: row.id,
        action: row.action,
        timestamp: row.timestamp,
        details: row.details,
        userName: row.user_name,
        userEmail: row.user_email,
    })) as ActivityLog[];

  } catch (error) {
    console.error("Database Error: Failed to fetch activity logs.", error);
    return [];
  }
}

/**
 * The server action that is called by the UI.
 * It now correctly calls the database function.
 */
export async function getActivityLogsAction(): Promise<ActivityLog[]> {
  try {
    const logs = await getActivityLogs();
    return logs;
  } catch (error) {
    console.error("Failed to fetch activity logs:", error);
    throw new Error('Failed to fetch activity logs.');
  }
}

/**
 * --- THIS IS THE FIX ---
 * The function has been renamed from 'logActivity' to 'addActivityLogAction'
 * to match what the ActivityLogContext is expecting to import.
 * It logs a new activity to the database.
 */
export async function addActivityLogAction(action: string, details: object = {}) {
    const userId = await getUserId();
    if(!userId) {
        console.warn("Could not log activity: user not found.");
        return;
    }

    try {
        await pool.query(
            `INSERT INTO activity_logs (user_id, action, details) VALUES ($1, $2, $3::jsonb)`,
            [userId, action, details]
        );
        revalidatePath('/admin/activity-logs'); // Ensure the log page updates
    } catch(error) {
        console.error("Failed to log activity:", error);
    }
}

