
'use server';

/**
 * @fileOverview AI flow to generate a draft reply to a prospect's email.
 *
 * - generateReplyDraft - A function that generates an email reply based on conversation context.
 * - GenerateReplyDraftInput - The input type for the generateReplyDraft function.
 * - GenerateReplyDraftOutput - The return type for the generateReplyDraft function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const GenerateReplyDraftInputSchema = z.object({
  prospectName: z.string().describe("The name of the prospect you are replying to."),
  prospectMessage: z.string().describe("The latest message from the prospect that you need to reply to."),
  threadContext: z.string().describe("The full context of the email thread so far, with each message prefixed by the sender's name (e.g., 'Me:' or 'Prospect Name:').")
});
export type GenerateReplyDraftInput = z.infer<typeof GenerateReplyDraftInputSchema>;

const GenerateReplyDraftOutputSchema = z.object({
  replyDraft: z.string().describe('The suggested email reply draft. It should be friendly, professional, and directly address the prospect\'s last message. Do not include a subject line or salutation (e.g., "Hi Prospect,").'),
});
export type GenerateReplyDraftOutput = z.infer<typeof GenerateReplyDraftOutputSchema>;

export async function generateReplyDraft(input: GenerateReplyDraftInput): Promise<GenerateReplyDraftOutput> {
  return generateReplyDraftFlow(input);
}

const generateReplyDraftPrompt = ai.definePrompt({
  name: 'generateReplyDraftPrompt',
  input: {schema: GenerateReplyDraftInputSchema},
  output: {schema: GenerateReplyDraftOutputSchema},
  prompt: `You are an expert sales development representative (SDR) tasked with writing a helpful and concise email reply to a prospect. Your goal is to keep the conversation going and move the prospect towards the next step (e.g., a meeting, a demo, providing more information).

Here is the full email thread for context:
---
{{{threadContext}}}
---

The prospect's name is {{{prospectName}}}.

The prospect's latest message, which you need to reply to, is: "{{{prospectMessage}}}"

Based on this, generate a draft for your reply.
- Be professional and friendly.
- Directly address their questions or comments from their last message.
- Keep the reply concise.
- Your response should only be the body of the email. Do not include a "Subject:" line and do not include the salutation (e.g., "Hi {{{prospectName}}},").

Return the reply draft in JSON format.`,
});

const generateReplyDraftFlow = ai.defineFlow(
  {
    name: 'generateReplyDraftFlow',
    inputSchema: GenerateReplyDraftInputSchema,
    outputSchema: GenerateReplyDraftOutputSchema,
  },
  async input => {
    const {output} = await generateReplyDraftPrompt(input);
    return output!;
  }
);
