import { Pool } from 'pg';

// This ensures that there is only one instance of the pool across your application.
// In a serverless environment, this helps manage connections efficiently.

// We declare the pool variable here but initialize it inside the conditional.
let pool: Pool;

// In a development environment, Next.js hot-reloads modules, which can create
// new Pool instances on every code change. To prevent this, we store the pool
// on the global object, which persists across reloads.
if (process.env.NODE_ENV === 'production') {
  // In production, we create a new pool instance.
  pool = new Pool({
    connectionString: process.env.DATABASE_URL,
    // Add SSL configuration for production databases (e.g., Vercel Postgres, Neon, etc.)
    ssl: {
      rejectUnauthorized: false,
    },
  });
} else {
  // We need to cast `globalThis` to `any` to attach our custom property.
  const globalWithPool = globalThis as typeof globalThis & {
    _pool?: Pool;
  };

  if (!globalWithPool._pool) {
    globalWithPool._pool = new Pool({
      connectionString: process.env.DATABASE_URL,
    });
  }
  pool = globalWithPool._pool;
}

// This is the crucial part: a named export of the 'pool' variable.
// The `import { pool } from '@/lib/db';` in your actions.ts will now work correctly.
export { pool };
