'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useCallback } from 'react';
import type { Template } from '@/lib/types';
import { getTemplates, addTemplate as addTemplateAction, updateTemplate as updateTemplateAction, deleteTemplate as deleteTemplateAction } from '@/app/(app)/templates/actions';

interface TemplateContextType {
  templates: Template[];
  addTemplate: (template: Omit<Template, 'id' | 'userId' | 'created_at' | 'last_modified'>) => Promise<any>;
  updateTemplate: (id: number, updatedTemplate: Partial<Omit<Template, 'id' | 'userId'>>) => Promise<any>;
  deleteTemplate: (id: number) => Promise<any>;
  getTemplate: (id: number) => Template | undefined;
  isLoading: boolean;
  error: string | null;
}

const TemplateContext = createContext<TemplateContextType | undefined>(undefined);

export const TemplateProvider = ({ children }: { children: ReactNode }) => {
  const [templates, setTemplates] = useState<Template[]>([]);
  const [isLoading, setIsLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  const fetchTemplates = useCallback(async () => {
    setIsLoading(true);
    setError(null);
    try {
      const templatesData = await getTemplates();
      setTemplates(templatesData);
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to fetch templates';
      console.error('Error fetching templates:', errorMessage);
      setError(errorMessage);
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchTemplates();
  }, [fetchTemplates]);

  const addTemplate = async (templateData: Omit<Template, 'id' | 'userId' | 'created_at' | 'last_modified'>) => {
    try {
      const result = await addTemplateAction(templateData);
      if (result.success) {
        await fetchTemplates(); // Re-fetch to get the source of truth
      }
      return result;
    } catch (error) {
      setError('Failed to add template');
      throw error;
    }
  };

  const updateTemplate = async (id: number, updatedTemplateData: Partial<Omit<Template, 'id' | 'userId'>>) => {
    try {
      const result = await updateTemplateAction(id, updatedTemplateData);
      if (result.success) {
        await fetchTemplates();
      }
      return result;
    } catch (error) {
      setError('Failed to update template');
      throw error;
    }
  };

  const deleteTemplate = async (id: number) => {
    try {
      const result = await deleteTemplateAction(id);
      if (result.success) {
        await fetchTemplates();
      }
      return result;
    } catch (error) {
      setError('Failed to delete template');
      throw error;
    }
  };

  const getTemplate = (id: number) => {
      return templates.find(t => t.id === id);
  }

  return (
    <TemplateContext.Provider value={{ templates, addTemplate, updateTemplate, deleteTemplate, getTemplate, isLoading, error }}>
      {children}
    </TemplateContext.Provider>
  );
};

export const useTemplates = (): TemplateContextType => {
  const context = useContext(TemplateContext);
  if (context === undefined) {
    throw new Error('useTemplates must be used within a TemplateProvider');
  }
  return context;
};
