
'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useCallback } from 'react';
import type { AudienceList, Prospect } from '@/lib/types';
import { getAudienceLists, addAudienceList, deleteProspectFromDb, addProspectToDb } from '@/app/(app)/audience/actions';
import { getCurrentUser } from '@/app/(app)/admin/actions';

interface AudienceContextType {
  audienceLists: AudienceList[];
  addAudienceList: (list: Omit<AudienceList, 'id' | 'userId' | 'created_at' | 'last_modified'>, userId: string) => Promise<void>;
  updateAudienceList: (id: string, updatedList: Partial<AudienceList>) => Promise<void>;
  deleteAudienceList: (id: string, userId: string) => Promise<void>;
  getAudienceList: (id: string) => Promise<AudienceList | undefined>;
  fetchAudienceLists: () => Promise<void>;
  updateProspect: (prospectId: string, sourceListId: string, newProspectData: Partial<Prospect>, newListId?: string) => Promise<void>;
  deleteProspect: (prospectId: string, sourceListId: string) => Promise<void>;
  addProspectToList: (listId: string, prospect: Prospect) => Promise<void>;
}

const AudienceContext = createContext<AudienceContextType | undefined>(undefined);

export const AudienceProvider = ({ children }: { children: ReactNode }) => {
  const [audienceLists, setAudienceLists] = useState<AudienceList[]>([]);
  const [currentUserId, setCurrentUserId] = useState<string | null>(null);

  useEffect(() => {
    const fetchUserAndSetId = async () => {
      const email = sessionStorage.getItem('userEmail');
      if (email) {
        const user = await getCurrentUser(email);
        if (user) {
          setCurrentUserId(user.id);
        }
      }
    };
    fetchUserAndSetId();
  }, []);

  const fetchAudienceLists = useCallback(async () => {
    let userIdToFetch = currentUserId;
    if (!userIdToFetch) {
        const email = sessionStorage.getItem('userEmail');
        if (email) {
            const user = await getCurrentUser(email);
            if (user) {
                userIdToFetch = user.id;
            }
        }
    }
    
    if (userIdToFetch) {
      const fetchedLists = await getAudienceLists(userIdToFetch);
      setAudienceLists(fetchedLists);
    } else {
        setAudienceLists([]);
    }
  }, [currentUserId]);

  useEffect(() => {
    fetchAudienceLists();
  }, [fetchAudienceLists]);

  const addAudienceListAction = async (list: Omit<AudienceList, 'id' | 'userId' | 'created_at' | 'last_modified'>, userId: string) => {
    if (!userId) throw new Error("User not authenticated");
    const now = new Date().toISOString();
    const newList: AudienceList = {
        ...list,
        id: `LIST-${Date.now()}`,
        userId: userId,
        created_at: now,
        last_modified: now,
    };
    await addAudienceList(newList, userId);
    await fetchAudienceLists();
  };

  const updateAudienceListAction = async (id: string, updatedList: Partial<AudienceList>) => {
    const userId = currentUserId || sessionStorage.getItem('userId');
    if (!userId) throw new Error("User not authenticated");
    const listToUpdate = audienceLists.find(l => l.id === id);
    if (listToUpdate) {
        const finalUpdate = { ...updatedList, last_modified: new Date().toISOString() };
        await updateAudienceList(id, userId, finalUpdate);
        await fetchAudienceLists();
    }
  };

  const deleteAudienceListAction = async (id: string, userId: string) => {
    if (!userId) throw new Error("User not authenticated");
    await deleteAudienceList(id, userId);
    setAudienceLists(prevLists => prevLists.filter(list => list.id !== id));
  };

  const getAudienceListAction = useCallback(async (id: string): Promise<AudienceList | undefined> => {
    const userId = currentUserId || sessionStorage.getItem('userId');
    if (!userId) return undefined;
    
    const localList = audienceLists.find(l => l.id === id);
    if (localList) {
      return localList;
    }
    
    try {
      const dbList = await getAudienceListById(id, userId);
      if (dbList) {
        setAudienceLists(prev => {
            const listExists = prev.some(l => l.id === dbList.id);
            if(listExists) {
                return prev.map(l => l.id === dbList.id ? dbList : l);
            }
            return [...prev, dbList];
        });
        return dbList;
      }
    } catch (error) {
      console.error('Error fetching audience list:', error);
    }
    
    return undefined;
  }, [audienceLists, currentUserId]);
  
  const deleteProspect = async (prospectId: string, sourceListId: string) => {
    const userId = currentUserId || sessionStorage.getItem('userId');
    if (!userId) throw new Error("User not authenticated");
    await deleteProspectFromDb(prospectId, sourceListId, userId);
    await fetchAudienceLists();
  };

  const addProspectToList = async (listId: string, prospect: Prospect) => {
      const userId = currentUserId || sessionStorage.getItem('userId');
      if (!userId) throw new Error("User not authenticated");
      await addProspectToDb(listId, userId, prospect);
      await fetchAudienceLists();
  }

  const updateProspect = async (prospectId: string, sourceListId: string, newProspectData: Partial<Prospect>, newListId?: string) => {
      // If the list is not changing, just update the prospect data in the current list
      if (!newListId || newListId === sourceListId) {
          const sourceList = audienceLists.find(l => l.id === sourceListId);
          if (sourceList && sourceList.prospects) {
              const updatedProspects = sourceList.prospects.map(p => p.id === prospectId ? {...p, ...newProspectData} as Prospect : p);
              await updateAudienceListAction(sourceListId, { prospects: updatedProspects });
          }
      } else {
          const sourceList = audienceLists.find(l => l.id === sourceListId);
          const destinationList = audienceLists.find(l => l.id === newListId);

          if (sourceList && destinationList && sourceList.prospects) {
              const prospectToMove = sourceList.prospects.find(p => p.id === prospectId);
              if (!prospectToMove) return;

              // 1. Remove from source list
              const prospectsInSource = sourceList.prospects.filter(p => p.id !== prospectId);
              // 2. Add updated prospect to destination list
              const prospectsInDest = [...(destinationList.prospects || []), { ...prospectToMove, ...newProspectData } as Prospect];
              
              // 3. Update both lists in parallel
              await Promise.all([
                  updateAudienceListAction(sourceListId, { prospects: prospectsInSource, count: prospectsInSource.length }),
                  updateAudienceListAction(newListId, { prospects: prospectsInDest, count: prospectsInDest.length })
              ]);
          }
      }
  };


  return (
    <AudienceContext.Provider value={{ 
        audienceLists, 
        addAudienceList: addAudienceListAction, 
        updateAudienceList: updateAudienceListAction, 
        deleteAudienceList: deleteAudienceListAction, 
        getAudienceList: getAudienceListAction, 
        fetchAudienceLists, 
        updateProspect, 
        deleteProspect,
        addProspectToList
    }}>
      {children}
    </AudienceContext.Provider>
  );
};

export const useAudience = (): AudienceContextType => {
  const context = useContext(AudienceContext);
  if (context === undefined) {
    throw new Error('useAudience must be used within an AudienceProvider');
  }
  return context;
};
