'use client';

import { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { ScrollArea } from "@/components/ui/scroll-area";
import { CampaignForm, CampaignFormState, CampaignFormHandlers } from "./campaign-form"; // Import the unified form
import type { Campaign, AudienceList, SequenceStep } from "@/lib/types";
import { Button } from './ui/button';
// --- NEW IMPORTS: We need both translator functions ---
import { mapCampaignToFormState, mapFormStateToCampaignData } from '@/lib/campaignUtils';

// The initial state for an empty form, used for resetting
const initialState: CampaignFormState = {
  name: '',
  objective: '',
  selectedAudience: null,
  sequence: [],
  senderIds: [],
  trackingDomainId: undefined,
  warmup: false,
  throttling: false,
};

export function EditCampaignDialog({ campaign, open, onOpenChange, onSave }: { campaign: Campaign | null, open: boolean, onOpenChange: (open: boolean) => void, onSave: (campaign: Partial<Campaign>, newStatus?: Campaign['status']) => void }) {
  // --- THIS IS THE KEY: State is now managed locally, not from a shared context ---
  const [formState, setFormState] = useState<CampaignFormState>(initialState);

  useEffect(() => {
    // When the dialog opens with a campaign, use our new "translator" to safely load the data
    if (open && campaign) {
      const loadedState = mapCampaignToFormState(campaign);
      setFormState(loadedState);
    }
  }, [open, campaign]);

  const handleOpenChange = (isOpen: boolean) => {
    if (!isOpen) {
      setFormState(initialState); // Reset state when the dialog closes
    }
    onOpenChange(isOpen);
  };

  const handleSave = (status?: Campaign['status']) => {
    if (campaign) {
      try {
        // Use our "translator" to prepare the data for saving
        const campaignDataToSave = mapFormStateToCampaignData(formState, status || campaign.status);
        onSave({ ...campaign, ...campaignDataToSave }, status);
        onOpenChange(false);
      } catch (error) {
        // Handle error if audience is missing, etc.
        console.error("Failed to save campaign:", error);
      }
    }
  };

  // Create the handlers to pass down to the reusable form component
  const handlers: CampaignFormHandlers = {
      setFormField: (field, value) => setFormState(prev => ({ ...prev, [field]: value })),
      setSelectedAudience: (list) => setFormState(prev => ({ ...prev, selectedAudience: list })),
      setSequence: (sequence) => setFormState(prev => ({...prev, sequence})),
      setSenderIds: (ids) => setFormState(prev => ({...prev, senderIds: ids})),
  };

  if (!campaign) return null;

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogContent className="max-w-4xl h-[90vh] flex flex-col">
        <DialogHeader>
          <DialogTitle>Edit Campaign: {campaign.name}</DialogTitle>
          <DialogDescription>
            Modify the details of your campaign below.
          </DialogDescription>
        </DialogHeader>
        <ScrollArea className="flex-grow pr-6 -mr-6">
          <CampaignForm mode="edit" formState={formState} handlers={handlers} />
        </ScrollArea>
        <DialogFooter className="pt-4 border-t flex-shrink-0">
            <Button variant="ghost" onClick={() => onOpenChange(false)}>Cancel</Button>
            <Button variant="outline" onClick={() => handleSave()}>Save Changes</Button>
            <Button onClick={() => handleSave('Scheduled')}>Save & Schedule</Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

