'use client';

import { useState, useEffect, useMemo } from "react";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { 
  PlusCircle, 
  Trash2,
  FileQuestion,
  Users,
  Send,
  Loader2,
  Search,
  X,
  Save,
  UserPlus,
  ArrowLeft
} from "lucide-react";
import { useAudience } from "@/context/AudienceContext";
import type { AudienceList, AudienceListStatus, CampaignObjective, SenderInfra, Prospect } from "@/lib/types";
import { useToast } from "@/hooks/use-toast";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";

function AddProspectDialog({ onAddProspect }: { onAddProspect: (prospect: Prospect) => void }) {
    const [open, setOpen] = useState(false);
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const { toast } = useToast();

    const handleSubmit = () => {
        if (!name || !email) {
            toast({ variant: 'destructive', title: 'Missing fields', description: 'Please enter both name and email.' });
            return;
        }
        const newProspect: Prospect = {
            id: `PROS${Date.now()}`,
            name,
            email,
        };
        onAddProspect(newProspect);
        setName('');
        setEmail('');
        setOpen(false);
        toast({ title: 'Prospect Added', description: `${name} has been added to the list.` });
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                <Button><UserPlus className="mr-2 h-4 w-4" /> Add Prospect</Button>
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>Add New Prospect</DialogTitle>
                    <DialogDescription>Manually add a new prospect to this audience list.</DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4">
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="prospect-name" className="text-right">Name</Label>
                        <Input id="prospect-name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="prospect-email" className="text-right">Email</Label>
                        <Input id="prospect-email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" />
                    </div>
                </div>
                <DialogFooter>
                    <Button type="button" onClick={handleSubmit}>Add Prospect</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

export function EditAudienceDialog({ list, open, onOpenChange }: { list: AudienceList | null, open: boolean, onOpenChange: (open: boolean) => void }) {
  const router = useRouter();
  const { updateAudienceList, deleteAudienceList, deleteProspect, addProspectToList } = useAudience();
  const { toast } = useToast();

  const [currentList, setCurrentList] = useState<AudienceList | null>(null);
  const [prospectSearchTerm, setProspectSearchTerm] = useState('');

  useEffect(() => {
    if (list) {
        setCurrentList(JSON.parse(JSON.stringify(list))); // Deep copy
    }
    // Reset search on dialog open/list change
    setProspectSearchTerm('');
  }, [list, open]);

  const filteredProspects = useMemo(() => {
    if (!currentList?.prospects) return [];
    if (!prospectSearchTerm) return currentList.prospects;
    return currentList.prospects.filter(p => 
        p.name.toLowerCase().includes(prospectSearchTerm.toLowerCase()) ||
        p.email.toLowerCase().includes(prospectSearchTerm.toLowerCase())
    );
  }, [currentList?.prospects, prospectSearchTerm]);


  const handleUpdateName = (name: string) => {
    if (currentList) {
        setCurrentList({ ...currentList, name });
    }
  };
  
  const handleAddProspect = async (prospect: Prospect) => {
      if (currentList) {
          await addProspectToList(currentList.id, prospect);
          setCurrentList(prev => prev ? { ...prev, prospects: [...(prev.prospects || []), prospect], count: (prev.count || 0) + 1} : null);
      }
  }

  const handleRemoveProspect = async (prospectId: string) => {
      if(currentList && currentList.prospects) {
          await deleteProspect(prospectId, currentList.id);
          const updatedProspects = currentList.prospects.filter(p => p.id !== prospectId);
          setCurrentList({ 
              ...currentList, 
              prospects: updatedProspects, 
              count: updatedProspects.length 
          });
          toast({ title: "Prospect Removed" });
      }
  }

  const handleSaveChanges = async () => {
    if (currentList) {
        if (!currentList.name) {
            toast({
                variant: "destructive",
                title: "Missing Information",
                description: "Audience list name is required.",
            });
            return;
        }
      
      await updateAudienceList(currentList.id, { name: currentList.name });

      toast({
        title: `Audience List Updated`,
        description: `Changes to "${currentList.name}" have been saved.`,
      });
      onOpenChange(false);
    }
  };

  const handleDelete = async () => {
    if (currentList) {
        await deleteAudienceList(currentList.id, currentList.userId);
        toast({
            title: "Audience List Deleted",
            description: `List "${currentList.name}" has been deleted.`,
        });
        onOpenChange(false);
    }
  }
  
  if (!currentList) {
      return null;
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
    <DialogContent className="max-w-4xl h-[90vh] flex flex-col">
      <DialogHeader>
        <DialogTitle>Editing: {currentList.name}</DialogTitle>
        <div className="flex justify-between items-center pt-4">
             <div>
                <AlertDialog>
                    <AlertDialogTrigger asChild>
                        <Button type="button" variant="destructive">
                            <Trash2 className="mr-2 h-4 w-4" />
                            Delete List
                        </Button>
                    </AlertDialogTrigger>
                    <AlertDialogContent>
                        <AlertDialogHeader>
                        <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                        <AlertDialogDescription>
                            This action cannot be undone. This will permanently delete this audience list.
                        </AlertDialogDescription>
                        </AlertDialogHeader>
                        <AlertDialogFooter>
                        <AlertDialogCancel>Cancel</AlertDialogCancel>
                        <AlertDialogAction onClick={handleDelete}>Continue</AlertDialogAction>
                        </AlertDialogFooter>
                    </AlertDialogContent>
                </AlertDialog>
            </div>
           
            <div className="flex gap-2">
                 <Button variant="outline" onClick={() => onOpenChange(false)}>
                    <ArrowLeft className="mr-2 h-4 w-4" />
                    Back
                </Button>
                <Button type="button" onClick={handleSaveChanges}>
                    <Save className="mr-2 h-4 w-4" /> Save Changes
                </Button>
            </div>
        </div>
      </DialogHeader>

      <div className="space-y-8 overflow-y-auto flex-1 pr-6 -mr-6 py-4">
        <Card>
          <CardHeader>
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary">
                <FileQuestion className="h-6 w-6" />
              </div>
              <div>
                <CardTitle>Details</CardTitle>
                <CardDescription>
                  Define the name for this audience list.
                </CardDescription>
              </div>
            </div>
          </CardHeader>
          <CardContent className="grid md:grid-cols-2 gap-6">
            <div className="grid gap-2">
              <Label htmlFor="list-name">List Name</Label>
              <Input 
                id="list-name" 
                value={currentList.name}
                onChange={(e) => handleUpdateName(e.target.value)}
              />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary">
                <Users className="h-6 w-6" />
              </div>
              <div>
                <CardTitle>Prospects</CardTitle>
                <CardDescription>
                  Manage the clients in this list. Current count: {currentList.prospects?.length ?? 0}
                </CardDescription>
              </div>
               <div className="ml-auto">
                    <AddProspectDialog onAddProspect={handleAddProspect} />
               </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="mb-4">
                <div className="relative">
                    <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                    <Input
                        placeholder="Search prospects in this list..."
                        className="pl-10"
                        value={prospectSearchTerm}
                        onChange={(e) => setProspectSearchTerm(e.target.value)}
                    />
                </div>
            </div>
             <Table>
                <TableHeader>
                    <TableRow>
                        <TableHead className="w-[80px]">Sr. No.</TableHead>
                        <TableHead>Name</TableHead>
                        <TableHead>Email</TableHead>
                        <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {filteredProspects.length > 0 ? (
                        filteredProspects.map((prospect, index) => (
                            <TableRow key={prospect.id}>
                                <TableCell>{index + 1}</TableCell>
                                <TableCell>{prospect.name}</TableCell>
                                <TableCell>{prospect.email}</TableCell>
                                <TableCell className="text-right">
                                    <Button type="button" variant="ghost" size="icon" onClick={() => handleRemoveProspect(prospect.id)}><Trash2 className="h-4 w-4" /></Button>
                                </TableCell>
                            </TableRow>
                        ))
                    ) : (
                        <TableRow>
                            <TableCell colSpan={4} className="h-24 text-center text-muted-foreground">
                                {prospectSearchTerm ? "No prospects match your search." : "No prospects in this list."}
                            </TableCell>
                        </TableRow>
                     )}
                </TableBody>
            </Table>
          </CardContent>
        </Card>
      </div>

    </DialogContent>
    </Dialog>
  );
}
