'use client';

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { PlusCircle, Trash2, FileQuestion, Users, Mails, Send, Flame, Gauge, ChevronDown, Globe, Book } from "lucide-react";
import type { Campaign, CampaignObjective, AudienceList, SequenceStep, Template } from "@/lib/types";
import { useAudience } from "@/context/AudienceContext";
import { useTemplates } from "@/context/TemplateContext";
import { Badge } from "@/components/ui/badge";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Separator } from "@/components/ui/separator";
import { getAvailableSenders } from '@/app/(app)/campaigns/actions';
import { getVerifiedTrackingDomains } from '@/app/(app)/settings/domains/actions';
import { SenderSelectionDialog } from '@/app/(app)/campaigns/_components/sender-selection-dialog';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog";




// Define the state shape that this form expects
export interface CampaignFormState {
  name: string;
  objective: CampaignObjective | '';
  selectedAudience: AudienceList | null;
  sequence: SequenceStep[];
  senderIds: number[];
  trackingDomainId?: number;
  warmup: boolean;
  throttling: boolean;
}

// Define the handlers the form needs to function
interface CampaignFormHandlers {
  setFormField: <K extends keyof CampaignFormState>(field: K, value: CampaignFormState[K]) => void;
  setSelectedAudience: (list: AudienceList | null) => void;
  setSequence: (sequence: SequenceStep[]) => void;
  setSenderIds: (ids: number[]) => void;
}

type Sender = { id: number; name: string; service_type: string };
type VerifiedDomain = { id: number; domain_name: string };

function SelectAudienceDropdown({ onSelect, onNew }: { onSelect: (list: AudienceList) => void, onNew: () => void }) {
    const { audienceLists } = useAudience();
    const savedLists = audienceLists.filter(list => list.status === 'Saved');
    return (
      <DropdownMenu>
        <DropdownMenuTrigger asChild><Button variant="outline">Select/Change Audience <ChevronDown className="ml-2 h-4 w-4" /></Button></DropdownMenuTrigger>
        <DropdownMenuContent align="start" className="w-80">
          {savedLists.map(list => ( <DropdownMenuItem key={list.id} onClick={() => onSelect(list)}><div><span>{list.name}</span><span className="text-xs text-muted-foreground">~{list.count.toLocaleString()} recipients</span></div></DropdownMenuItem> ))}
          <DropdownMenuSeparator />
          <DropdownMenuItem onClick={onNew}><PlusCircle className="mr-2 h-4 w-4" />Create New Audience</DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
    )
}

function SelectTemplateDialog({ onSelectTemplate, open, onOpenChange }: { onSelectTemplate: (template: Template) => void, open: boolean, onOpenChange: (open: boolean) => void }) {
    const { templates } = useTemplates();
    const handleSelect = (template: Template) => { onSelectTemplate(template); onOpenChange(false); };
    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl"><DialogHeader><DialogTitle>Select an Email Template</DialogTitle><DialogDescription>Choose a starting point for your email.</DialogDescription></DialogHeader>
                <div className="py-4 max-h-[60vh] overflow-y-auto"><div className="space-y-2">
                    {templates.map(template => ( <button key={template.id} onClick={() => handleSelect(template)} className="w-full text-left p-3 border rounded-lg hover:bg-muted/50"><p className="font-semibold">{template.name}</p><p className="text-sm text-muted-foreground truncate">Subject: {template.subject}</p></button>))}
                    {templates.length === 0 && ( <p className="text-center text-muted-foreground py-8">No templates found.</p> )}
                </div></div>
            </DialogContent>
        </Dialog>
    );
}

// This is the reusable form component. It is "dumb" and only displays the UI.
export function CampaignForm({ formState, handlers, mode }: { formState: CampaignFormState, handlers: CampaignFormHandlers, mode: 'create' | 'edit' }) {
  const { name, objective, warmup, throttling, selectedAudience, sequence = [], senderIds = [], trackingDomainId } = formState;
  const { setFormField, setSelectedAudience, setSequence, setSenderIds } = handlers;

  const router = useRouter();

  const [availableSenders, setAvailableSenders] = useState<Sender[]>([]);
  const [verifiedDomains, setVerifiedDomains] = useState<VerifiedDomain[]>([]);
  const [isSenderDialogOpen, setIsSenderDialogOpen] = useState(false);
  const [templateDialogState, setTemplateDialogState] = useState<{ open: boolean; stepId: string | null }>({ open: false, stepId: null });

  useEffect(() => {
    getAvailableSenders().then(setAvailableSenders);
    getVerifiedTrackingDomains().then(setVerifiedDomains);
  }, []);

  const addSequenceStep = () => setSequence([...sequence, { id: `step${Date.now()}`, name: `Follow-up ${sequence.length}`, description: 'A follow-up email.', delay: 3, condition: 'not-opened', templateId: null, subject: '', body: '' }]);
  const removeSequenceStep = (id: string) => {
    if (sequence.length > 1) setSequence(sequence.filter(step => step.id !== id));
  };
  const handleUpdateStep = (id: string, field: keyof SequenceStep, value: any) => setSequence(sequence.map(step => (step.id === id ? { ...step, [field]: value } : step)));
  const handleSelectTemplate = (template: Template) => {
    if (templateDialogState.stepId) setSequence(current => current.map(step => step.id === templateDialogState.stepId ? { ...step, templateId: template.id, name: step.name, subject: template.subject, body: template.body } : step));
    setTemplateDialogState({ open: false, stepId: null });
  };

  const selectedSenderObjects = availableSenders.filter(s => senderIds.includes(s.id));

  return (
    <>
      <div className="space-y-8 p-1">
        {/* Step 1 & 2 Cards... (Unchanged) */}
        <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><FileQuestion className="h-6 w-6" /></div><div><CardTitle>Step 1: Objective</CardTitle><CardDescription>Define the goal of your campaign.</CardDescription></div></div></CardHeader>
            <CardContent className="grid md:grid-cols-2 gap-6">
                <div className="grid gap-2"><Label htmlFor={`campaign-name-${mode}`}>Campaign Name</Label><Input id={`campaign-name-${mode}`} value={name} onChange={(e) => setFormField('name', e.target.value)} /></div>
                <div className="grid gap-2"><Label htmlFor={`campaign-objective-${mode}`}>Campaign Objective</Label><Select value={objective} onValueChange={(value) => setFormField('objective', value as CampaignObjective)}><SelectTrigger id={`campaign-objective-${mode}`}><SelectValue placeholder="Select an objective" /></SelectTrigger><SelectContent><SelectItem value="Lead Nurturing">Lead Nurturing</SelectItem><SelectItem value="Sales Conversion">Sales Conversion</SelectItem><SelectItem value="Engagement/Retention">Engagement/Retention</SelectItem><SelectItem value="Brand Awareness">Brand Awareness</SelectItem></SelectContent></Select></div>
            </CardContent>
        </Card>
        <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Users className="h-6 w-6" /></div><div><CardTitle>Step 2: Audience</CardTitle><CardDescription>Select who will receive this campaign.</CardDescription></div></div></CardHeader>
            <CardContent>
                {selectedAudience ? (<Card className="bg-muted/30"><CardHeader><div className="flex justify-between items-center"><div><CardTitle className="text-base">{selectedAudience.name}</CardTitle><CardDescription>~{selectedAudience.count.toLocaleString()} recipients</CardDescription></div><SelectAudienceDropdown onSelect={setSelectedAudience} onNew={() => router.push('/audience/new?redirect=/campaigns')}/></div></CardHeader></Card>) : (<div className="flex flex-col items-center justify-center text-center gap-4 p-8 border-2 border-dashed rounded-lg"><p className="text-muted-foreground">No audience list selected.</p><SelectAudienceDropdown onSelect={setSelectedAudience} onNew={() => router.push('/audience/new?redirect=/campaigns')}/></div>)}
            </CardContent>
        </Card>

        {/* --- THIS IS THE UPGRADED STEP 3 --- */}
        <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Mails className="h-6 w-6" /></div><div><CardTitle>Step 3: Email Sequence</CardTitle><CardDescription>Design the flow and content of your campaign.</CardDescription></div></div></CardHeader>
            <CardContent className="space-y-4">
              {sequence.map((step, index) => (
                  <div key={step.id} className="p-4 border rounded-lg space-y-4">
                      <div className="flex justify-between items-start"><div className="grid gap-1.5 flex-1"><Label htmlFor={`step-name-${mode}-${step.id}`}>Step Name</Label><Input id={`step-name-${mode}-${step.id}`} value={step.name} onChange={(e) => handleUpdateStep(step.id, 'name', e.target.value)} className="text-base font-semibold"/></div><Button variant="ghost" size="icon" onClick={() => removeSequenceStep(step.id)} className="ml-4"><Trash2 className="h-4 w-4" /></Button></div>
                      <Separator/>
                      <div className="space-y-2"><div className="flex justify-between items-center"><Label htmlFor={`step-subject-${mode}-${step.id}`}>Subject Line</Label><Button variant="link" size="sm" className="h-auto p-0" onClick={() => setTemplateDialogState({ open: true, stepId: step.id })}><Book className="mr-2 h-4 w-4" /> Load from Template</Button></div><Input id={`step-subject-${mode}-${step.id}`} value={step.subject || ''} onChange={(e) => handleUpdateStep(step.id, 'subject', e.target.value)} placeholder="Enter email subject..."/></div>
                      {/* The Textarea is now replaced with the RichTextEditor */}
                      <div className="space-y-2"><Label>Email Body</Label><RichTextEditor content={step.body || ''} onChange={(newContent) => handleUpdateStep(step.id, 'body', newContent)}/></div>
                      {index > 0 && (<div className="grid gap-2 pt-4 border-t"><Label>Logic</Label><div className="flex items-center gap-2 flex-wrap"><span>Send</span><Input type="number" value={step.delay} onChange={(e) => handleUpdateStep(step.id, 'delay', parseInt(e.target.value))} className="w-16 h-9"/><span>days after previous step, if recipient</span><Select value={step.condition} onValueChange={(v) => handleUpdateStep(step.id, 'condition', v)}><SelectTrigger className="w-[150px] h-9"><SelectValue/></SelectTrigger><SelectContent><SelectItem value="not-opened">has not opened</SelectItem><SelectItem value="not-clicked">has not clicked</SelectItem><SelectItem value="opened-no-click">opened but not clicked</SelectItem></SelectContent></Select></div></div>)}
                  </div>
              ))}
              <Button variant="outline" className="w-full" onClick={addSequenceStep}><PlusCircle className="mr-2 h-4 w-4"/>Add Step</Button>
            </CardContent>
        </Card>

        {/* Step 4 Card... (Unchanged) */}
        <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Send className="h-6 w-6" /></div><div><CardTitle>Step 4: Sender & Delivery</CardTitle><CardDescription>Configure how emails will be sent and tracked.</CardDescription></div></div></CardHeader>
            <CardContent className="grid md:grid-cols-2 gap-8"><div className="grid gap-4"><Label className="font-semibold">Sending Infrastructure</Label><p className="text-sm text-muted-foreground">Select one or more verified senders for rotation.</p><Button type="button" variant="outline" onClick={() => setIsSenderDialogOpen(true)} className="max-w-xs"><PlusCircle className="mr-2 h-4 w-4" />Select Senders ({senderIds.length} selected)</Button>{selectedSenderObjects.length > 0 && (<div className="p-3 border rounded-md space-y-2 bg-muted/50"><h4 className="text-xs font-medium text-muted-foreground">Selected:</h4><div className="flex flex-wrap gap-1.5">{selectedSenderObjects.map(sender => (<Badge key={sender.id} variant="secondary">{sender.name}</Badge>))}</div></div>)}</div><div className="grid gap-6"><div className="grid gap-2"><Label className="font-semibold flex items-center gap-2"><Globe className="h-4 w-4" /> Tracking Domain</Label><p className="text-sm text-muted-foreground">Choose a verified domain for tracking.</p><Select value={trackingDomainId?.toString()} onValueChange={(value) => setFormField('trackingDomainId', Number(value))} disabled={verifiedDomains.length === 0}><SelectTrigger><SelectValue placeholder="Select a tracking domain..." /></SelectTrigger><SelectContent>{verifiedDomains.map(domain => (<SelectItem key={domain.id} value={domain.id.toString()}>{domain.domain_name}</SelectItem>))}</SelectContent></Select>{verifiedDomains.length === 0 && <p className="text-xs text-red-500">No verified tracking domains found.</p>}</div><div className="flex items-start space-x-3 space-y-0 rounded-md border p-4"><Flame className="h-5 w-5 mt-1 text-muted-foreground"/><div className="space-y-1 leading-none"><div className="flex items-center gap-2"><Checkbox id={`warm-up-${mode}`} checked={warmup} onCheckedChange={(checked) => setFormField('warmup', !!checked)} /><Label htmlFor={`warm-up-${mode}`}>Enable Warm-up</Label></div></div></div><div className="flex items-start space-x-3 space-y-0 rounded-md border p-4"><Gauge className="h-5 w-5 mt-1 text-muted-foreground"/><div className="space-y-1 leading-none"><div className="flex items-center gap-2"><Checkbox id={`throttling-${mode}`} checked={throttling} onCheckedChange={(checked) => setFormField('throttling', !!checked)} /><Label htmlFor={`throttling-${mode}`}>Enable Throttling</Label></div></div></div></div></CardContent>
        </Card>
      </div>
      <SenderSelectionDialog open={isSenderDialogOpen} onOpenChange={setIsSenderDialogOpen} availableSenders={availableSenders} selectedSenderIds={senderIds} onSave={setSenderIds} />
      <SelectTemplateDialog open={templateDialogState.open} onOpenChange={(isOpen) => setTemplateDialogState({ open: isOpen, stepId: null })} onSelectTemplate={handleSelectTemplate} />
    </>
  );
}
