
'use client';

import React, { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { PageHeader } from "@/components/page-header";
import { Save, ArrowLeft } from "lucide-react";
import { useTemplates } from "@/context/TemplateContext";
import { useToast } from "@/hooks/use-toast";
import { getCurrentUser } from "@/app/(app)/admin/actions";


export default function ManualNewTemplatePage() {
    const router = useRouter();
    const { addTemplate } = useTemplates();
    const { toast } = useToast();
    const [currentUserId, setCurrentUserId] = useState<string | null>(null);

    const [name, setName] = useState('');
    const [subject, setSubject] = useState('');
    const [salutation, setSalutation] = useState('');
    const [body, setBody] = useState('');
    const [closing, setClosing] = useState('');

    useEffect(() => {
        const fetchUser = async () => {
            const email = sessionStorage.getItem('userEmail');
            if (email) {
                const user = await getCurrentUser(email);
                if (user) {
                    setCurrentUserId(user.id);
                }
            }
        };
        fetchUser();
    }, []);
    
    const handleSave = async () => {
        if (!name || !subject || !body) {
            toast({ variant: 'destructive', title: 'Missing Fields', description: 'Template Name, Subject, and Body are required.'});
            return;
        }

        if (!currentUserId) {
            toast({ variant: 'destructive', title: 'Authentication Error', description: 'Could not create template. User not found.'});
            return;
        }

        const fullBody = [salutation, body, closing].filter(Boolean).join('\n\n');

        await addTemplate({
            name,
            subject,
            body: fullBody,
            preheader: '',
            cta: '',
        });

        toast({ title: 'Template Created', description: `"${name}" has been saved.` });
        router.push('/templates');
    };

    return (
        <>
            <PageHeader title="Create New Template Manually" />
            <div className="space-y-6">
                <Card>
                    <CardHeader>
                        <CardTitle>Template Details</CardTitle>
                        <CardDescription>Fill in the core details of your template.</CardDescription>
                    </CardHeader>
                    <CardContent className="grid gap-6">
                        <div className="grid gap-2">
                            <Label htmlFor="template-name">Template Name</Label>
                            <Input id="template-name" value={name} onChange={(e) => setName(e.target.value)} placeholder="e.g., Q1 Follow-Up Email" />
                        </div>
                        <div className="grid gap-2">
                            <Label htmlFor="template-subject">Subject</Label>
                            <Input id="template-subject" value={subject} onChange={(e) => setSubject(e.target.value)} placeholder="Catchy subject line" />
                        </div>
                         <div className="grid gap-2">
                            <Label htmlFor="template-salutation">Salutation</Label>
                            <Textarea id="template-salutation" value={salutation} rows={2} onChange={(e) => setSalutation(e.target.value)} placeholder="e.g., Hi {{name}},"/>
                        </div>
                        <div className="grid gap-2">
                            <Label htmlFor="template-body">Body</Label>
                            <Textarea id="template-body" value={body} rows={8} onChange={(e) => setBody(e.target.value)} placeholder="Write your email content here..."/>
                        </div>
                        <div className="grid gap-2">
                            <Label htmlFor="template-closing">Closing</Label>
                            <Textarea id="template-closing" value={closing} rows={3} onChange={(e) => setClosing(e.target.value)} placeholder="e.g., Best,\nYour Name"/>
                        </div>
                    </CardContent>
                </Card>
            </div>
            
            <div className="flex justify-end items-center mt-8 gap-2">
                <Button variant="outline" onClick={() => router.push('/templates')}>
                    <ArrowLeft className="mr-2 h-4 w-4" />
                    Cancel
                </Button>
                <Button onClick={handleSave}>
                    <Save className="mr-2 h-4 w-4" />
                    Save Template
                </Button>
            </div>
        </>
    );
}
