'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import { z } from 'zod';
import { promises as dns } from 'dns';
import { getUserId } from '@/lib/auth';

/**
 * Fetches all tracking domains for the current user. (This function remains the same)
 */
export async function getTrackingDomains() {
  const userId = await getUserId();
  if (!userId) return [];
  try {
    const res = await pool.query('SELECT id, domain_name, status FROM tracking_domains WHERE user_id = $1 ORDER BY created_at DESC', [userId]);
    return res.rows;
  } catch (error) {
    console.error("Database Error: Failed to fetch tracking domains.", error);
    return [];
  }
}

/**
 * --- UPGRADED ---
 * Adds a new tracking domain and its user-provided CNAME target.
 */
export async function addTrackingDomain(prevState: any, formData: FormData) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: 'Authentication required.' };

  const schema = z.object({
    domainName: z.string().min(3, "Please enter a valid domain name.").toLowerCase(),
    cnameTarget: z.string().min(3, "Please enter a valid CNAME target.").toLowerCase(), // <-- NEW
  });
  const validated = schema.safeParse({
    domainName: formData.get('domainName'),
    cnameTarget: formData.get('cnameTarget'), // <-- NEW
  });

  if (!validated.success) return { success: false, message: 'Invalid form data. Both fields are required.' };

  try {
    // Save both the domain and its target to the database
    await pool.query(
      'INSERT INTO tracking_domains (user_id, domain_name, cname_target) VALUES ($1, $2, $3)',
      [userId, validated.data.domainName, validated.data.cnameTarget] // <-- NEW
    );
    revalidatePath('/settings/domains');
    return { success: true, message: 'Domain added successfully.' };
  } catch (error) {
    console.error("Database Error:", error);
    return { success: false, message: 'Failed to add domain. It may already exist.' };
  }
}

/**
 * Deletes a tracking domain. (This function remains the same)
 */
export async function deleteTrackingDomain(domainId: number) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: 'Authentication required.' };

  try {
    await pool.query('DELETE FROM tracking_domains WHERE id = $1 AND user_id = $2', [domainId, userId]);
    revalidatePath('/settings/domains');
    return { success: true, message: 'Domain deleted.' };
  } catch (error) {
    console.error("Database Error:", error);
    return { success: false, message: 'Failed to delete domain.' };
  }
}

/**
 * --- UPGRADED ---
 * Verifies a tracking domain against the target the user has saved in the database.
 */
export async function verifyDomain(domainName: string) { // <-- Simplified: no longer needs appDomain
  const userId = await getUserId();
  if (!userId) return { success: false, message: 'Authentication error.' };

  try {
    // 1. Fetch the user-provided target from our database
    const dbRes = await pool.query(
      'SELECT cname_target FROM tracking_domains WHERE domain_name = $1 AND user_id = $2',
      [domainName, userId]
    );
    const expectedTarget = dbRes.rows[0]?.cname_target;

    if (!expectedTarget) {
      return { success: false, message: "Could not find a saved CNAME target for this domain." };
    }

    // 2. Look up the actual public DNS record
    const cnameRecords = await dns.resolveCname(domainName);
    const actualTarget = cnameRecords[0];

    // 3. Compare the two
    if (actualTarget.toLowerCase() === expectedTarget.toLowerCase()) {
      await pool.query("UPDATE tracking_domains SET status = 'Verified' WHERE domain_name = $1 AND user_id = $2", [domainName, userId]);
      revalidatePath('/settings/domains');
      return { success: true, message: `Verified! Points to ${actualTarget}.` };
    } else {
      await pool.query("UPDATE tracking_domains SET status = 'Failed' WHERE domain_name = $1 AND user_id = $2", [domainName, userId]);
      revalidatePath('/settings/domains');
      return { success: false, message: `Failed. Public DNS points to "${actualTarget}", but your saved target is "${expectedTarget}".` };
    }
  } catch (error: any) {
    await pool.query("UPDATE tracking_domains SET status = 'Failed' WHERE domain_name = $1 AND user_id = $2", [domainName, userId]);
    revalidatePath('/settings/domains');
    if (error.code === 'ENOTFOUND' || error.code === 'ENODATA') {
      return { success: false, message: 'Failed. No public CNAME record found.' };
    }
    return { success: false, message: 'An unexpected error occurred.' };
  }
}

export async function getVerifiedTrackingDomains() {
  const userId = await getUserId();
  if (!userId) return [];
  try {
    const res = await pool.query(
      "SELECT id, domain_name FROM tracking_domains WHERE user_id = $1 AND status = 'Verified' ORDER BY domain_name ASC",
      [userId]
    );
    return res.rows;
  } catch (error) {
    console.error("Database Error: Failed to fetch verified domains.", error);
    return [];
  }
}