
'use client';

import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { Copy, KeyRound, DollarSign, FileText, ArrowLeft, Loader2 } from "lucide-react";
import { useEffect, useState } from "react";
import { useToast } from "@/hooks/use-toast";
import type { User } from "@/lib/types";
import { Label } from "@/components/ui/label";
import Link from "next/link";
import { getCurrentUser } from "@/app/(app)/admin/actions";

export default function ApiAccessPage() {
    const { toast } = useToast();
    const [currentUser, setCurrentUser] = useState<User | null>(null);
    const [isLoading, setIsLoading] = useState(true);

    useEffect(() => {
        const fetchUser = async () => {
            if (typeof window !== 'undefined') {
                const email = sessionStorage.getItem('userEmail');
                if (email) {
                    const user = await getCurrentUser(email);
                    setCurrentUser(user);
                }
            }
            setIsLoading(false);
        };
        fetchUser();
    }, []);

    const handleCopyKey = (key: string) => {
        navigator.clipboard.writeText(key);
        toast({ title: "Copied!", description: "API Key has been copied to your clipboard." });
    }
    
    const homeHref = currentUser?.role === 'Admin' ? '/admin' : '/campaigns';

    if (isLoading) {
       return (
            <div className="flex justify-center items-center h-64">
                <Loader2 className="w-8 h-8 animate-spin text-primary" />
            </div>
       );
    }
    
    if (!currentUser) {
       return (
            <Card>
                <CardContent className="p-6 text-center text-muted-foreground">
                   Could not load user data. Please try logging in again.
                </CardContent>
            </Card>
       )
    }

    return (
        <div className="flex flex-col h-full">
            <PageHeader
                title="API Access"
                description="Manage your API key and monitor your usage."
            />
            <div className="grid gap-8 flex-1">
                <Card>
                    <CardHeader>
                        <CardTitle>Your API Key</CardTitle>
                        <CardDescription>
                            Use this key to access the SmartMail AI API. Do not share it publicly.
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                       {currentUser.apiKey ? (
                            <div className="flex items-center justify-between p-4 border rounded-lg bg-muted/50">
                               <div className="flex items-center gap-4">
                                    <KeyRound className="h-6 w-6 text-muted-foreground" />
                                    <div>
                                        <p className="font-semibold">Default Key</p>
                                        <p className="font-mono text-sm text-muted-foreground">{currentUser.apiKey}</p>
                                    </div>
                               </div>
                                <Button variant="outline" size="icon" onClick={() => handleCopyKey(currentUser.apiKey)}>
                                    <Copy className="h-4 w-4" />
                                </Button>
                            </div>
                       ) : (
                            <div className="text-center py-8 text-muted-foreground">
                                You do not have an API key. Please contact support.
                            </div>
                       )}
                    </CardContent>
                </Card>

                 <Card>
                    <CardHeader>
                        <CardTitle>API Usage</CardTitle>
                        <CardDescription>
                            Your current usage for this billing cycle.
                        </CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        <div className="grid gap-6">
                             <div className="space-y-4 text-sm">
                                <div className="flex justify-between">
                                    <span>API Calls</span>
                                    <span className="font-mono">1,234 / 10,000</span>
                                </div>
                                <Separator />
                                <div className="flex justify-between">
                                    <span>Data Enrichment</span>
                                    <span className="font-mono">56 / 1,000</span>
                                </div>
                            </div>
                            <Separator />
                             <div className="space-y-2">
                                <Label>Billing (Current Billing Cycle)</Label>
                                <div className="flex items-center gap-2">
                                    <DollarSign className="h-6 w-6 text-muted-foreground"/>
                                    <p className="text-2xl font-bold">$199.99</p>
                                </div>
                            </div>
                        </div>
                        <Separator />
                        <div className="flex justify-start pt-4">
                            <Link href="/settings/api-docs">
                                <Button variant="outline">
                                    <FileText className="mr-2 h-4 w-4" />
                                    View API Docs
                                </Button>
                            </Link>
                        </div>
                    </CardContent>
                </Card>
            </div>
             <div className="flex justify-end mt-6">
                <Link href={homeHref}>
                    <Button variant="outline">
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Home
                    </Button>
                </Link>
            </div>
        </div>
    );
}
