

'use client';

import { useEffect, useState, useMemo, useCallback } from 'react';
import { PageHeader } from "@/components/page-header";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Search, Download, PlusCircle, Upload, MoreHorizontal, Edit, Trash2, ArrowDownAZ, ArrowUpAZ, Clock, ChevronDown } from "lucide-react";
import { getProspects } from './actions';
import type { ProspectSource, Prospect, AudienceList } from '@/lib/types';
import { Badge } from "@/components/ui/badge";
import { useToast } from '@/hooks/use-toast';
import { useAudience } from "@/context/AudienceContext";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useRouter } from 'next/navigation';
import { useDropzone } from 'react-dropzone';
import { cn } from '@/lib/utils';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from '@/components/ui/alert-dialog';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { format } from 'date-fns';
import { getCurrentUser } from '../admin/actions';

type SortConfig = {
    key: keyof ProspectSource | 'lastModified';
    direction: 'asc' | 'desc';
} | null;

function EditProspectDialog({ prospect, open, onOpenChange, onUpdate }: { prospect: ProspectSource | null, open: boolean, onOpenChange: (open: boolean) => void, onUpdate: (prospectId: string, sourceListId: string, newProspectData: Partial<Prospect>) => void }) {
    const { toast } = useToast();
    
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');

    useEffect(() => {
        if (prospect) {
            setName(prospect.name);
            setEmail(prospect.email);
        }
    }, [prospect]);

    if (!prospect) return null;

    const handleUpdate = async (e?: React.FormEvent) => {
        e?.preventDefault();
        if (name === prospect.name && email === prospect.email) {
            toast({ title: 'No Changes Detected', description: 'No changes were made to the prospect.' });
            onOpenChange(false);
            return;
        }
        const updatedProspectData = { id: prospect.id, name, email };
        await onUpdate(prospect.id, prospect.sourceId, updatedProspectData);
        toast({ title: 'Prospect Updated', description: `${name}'s details have been saved.` });
        onOpenChange(false);
    };

    const isOneOff = prospect.sourceType === 'One-Off';
    const sourceText = isOneOff ? 'One-Off' : prospect.source;

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent>
                <form onSubmit={handleUpdate}>
                    <DialogHeader>
                        <DialogTitle>Edit Prospect</DialogTitle>
                        <DialogDescription>Update the details for {prospect.name}.</DialogDescription>
                    </DialogHeader>
                    <div className="grid gap-4 py-4">
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="prospect-name" className="text-right">Name</Label>
                            <Input id="prospect-name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" />
                        </div>
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="prospect-email" className="text-right">Email</Label>
                            <Input id="prospect-email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" />
                        </div>
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label className="text-right">Source</Label>
                             <div className="col-span-3">
                                <Badge variant={isOneOff ? 'secondary' : 'outline'}>{sourceText}</Badge>
                            </div>
                        </div>
                    </div>
                    <DialogFooter>
                        <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
                        <Button type="submit">Save Changes</Button>
                    </DialogFooter>
                </form>
            </DialogContent>
        </Dialog>
    );
}

function AddProspectDialog() {
    const { toast } = useToast();
    const { addAudienceList } = useAudience();
    const [open, setOpen] = useState(false);
    const [currentUserId, setCurrentUserId] = useState<string | null>(null);

    const [name, setName] = useState('');
    const [email, setEmail] = useState('');

    useEffect(() => {
        const fetchUser = async () => {
            const userEmail = sessionStorage.getItem('userEmail');
            if(userEmail) {
                const user = await getCurrentUser(userEmail);
                if (user) {
                    setCurrentUserId(user.id);
                }
            }
        };
        fetchUser();
    }, []);

    const resetState = () => {
        setName('');
        setEmail('');
    };

    const handleAddProspect = async (e?: React.FormEvent) => {
        e?.preventDefault();
        if (!name || !email) {
            toast({ variant: 'destructive', title: 'Missing fields', description: 'Please enter prospect name and email.' });
            return;
        }
        if (!currentUserId) {
            toast({ variant: 'destructive', title: 'Error', description: 'Could not identify user.' });
            return;
        }

        const now = new Date();
        const newProspect: Prospect = { id: `PROS-${now.getTime()}`, name, email };
        
        await addAudienceList({
            name: `Audience for ${newProspect.name}`,
            status: 'Saved',
            source: 'One-Off',
            count: 1,
            prospects: [newProspect]
        }, currentUserId);
        
        toast({ title: 'Prospect Added', description: `${name} has been added as a one-off contact.` });
        
        resetState();
        setOpen(false);
    };

    return (
        <Dialog open={open} onOpenChange={(isOpen) => {
            setOpen(isOpen);
            if (!isOpen) resetState();
        }}>
            <DialogTrigger asChild>
                <Button>
                    <PlusCircle className="mr-2 h-4 w-4" />
                    Add Prospect
                </Button>
            </DialogTrigger>
            <DialogContent>
                <form onSubmit={handleAddProspect}>
                    <DialogHeader>
                        <DialogTitle>Add a New Prospect</DialogTitle>
                        <DialogDescription>
                        This will add the contact as a one-off prospect. To add multiple prospects, use the Import List feature.
                        </DialogDescription>
                    </DialogHeader>
                    <div className="grid gap-4 py-4">
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="prospect-name" className="text-right">Name</Label>
                            <Input id="prospect-name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" />
                        </div>
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="prospect-email" className="text-right">Email</Label>
                            <Input id="prospect-email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" />
                        </div>
                    </div>
                    <DialogFooter>
                        <Button type="button" variant="outline" onClick={() => setOpen(false)}>Cancel</Button>
                        <Button type="submit">Add Prospect</Button>
                    </DialogFooter>
                </form>
            </DialogContent>
        </Dialog>
    );
}

const parseCsvLine = (line: string): string[] => {
    const result: string[] = [];
    let current = '';
    let inQuotes = false;
    for (let i = 0; i < line.length; i++) {
        const char = line[i];
        if (char === '"') {
            inQuotes = !inQuotes;
        } else if (char === ',' && !inQuotes) {
            result.push(current.trim());
            current = '';
        } else {
            current += char;
        }
    }
    result.push(current.trim());
    return result.map(val => val.replace(/^"|"$/g, ''));
};


function ImportProspectsDialog() {
    const { toast } = useToast();
    const { audienceLists, updateAudienceList, addAudienceList } = useAudience();
    const [open, setOpen] = useState(false);
    const [uploadedFile, setUploadedFile] = useState<File | null>(null);
    const [sourceType, setSourceType] = useState<'new' | 'existing'>('new');
    const [newListName, setNewListName] = useState('');
    const [existingListId, setExistingListId] = useState('');
    const [prospects, setProspects] = useState<Prospect[]>([]);
    const router = useRouter();
    const [currentUserId, setCurrentUserId] = useState<string | null>(null);

    useEffect(() => {
        const fetchUser = async () => {
            const email = sessionStorage.getItem('userEmail');
            if(email) {
                const user = await getCurrentUser(email);
                if (user) {
                    setCurrentUserId(user.id);
                }
            }
        };
        fetchUser();
    }, []);

    const resetState = () => {
        setUploadedFile(null);
        setSourceType('new');
        setNewListName('');
        setExistingListId('');
        setProspects([]);
    };

    const parseCSV = (fileContent: string): Prospect[] => {
        const lines = fileContent.trim().split('\n');
        const headers = lines[0].split(',').map(h => h.trim().toLowerCase().replace(/"/g, ''));
        
        const nameIndex = headers.indexOf('full name');
        const emailIndex = headers.indexOf('email');

        if(nameIndex === -1 || emailIndex === -1) {
            toast({ variant: 'destructive', title: 'Invalid CSV', description: 'CSV must contain "Full Name" and "Email" headers.' });
            return [];
        }

        return lines.slice(1).map(line => {
            const values = parseCsvLine(line);
            
            return {
                id: `PROS-CSV-${Date.now()}-${Math.random()}`,
                name: (values[nameIndex] || '').trim(),
                email: (values[emailIndex] || '').trim(),
            }
        }).filter(p => p.name && p.email);
    };

    const onDrop = useCallback((acceptedFiles: File[]) => {
        if (acceptedFiles.length > 0) {
            const file = acceptedFiles[0];
            const reader = new FileReader();
            reader.onload = (event) => {
                const content = event.target?.result as string;
                const parsedProspects = parseCSV(content);
                if(parsedProspects.length > 0) {
                    setUploadedFile(file);
                    setProspects(parsedProspects);
                    toast({ title: 'File Parsed', description: `${parsedProspects.length} prospects found.` });
                }
            };
            reader.readAsText(file);
        }
    }, [toast]);

    const { getRootProps, getInputProps, isDragActive, open: openFileDialog } = useDropzone({
        onDrop,
        accept: { 'text/csv': ['.csv'] },
        maxFiles: 1,
        noClick: true,
        noKeyboard: true,
    });

    const handleImport = async () => {
        if (!currentUserId) {
             toast({ variant: 'destructive', title: 'Error', description: 'Could not identify user.' });
             return;
        }

        if (prospects.length === 0) {
            toast({ variant: 'destructive', title: 'No Prospects', description: 'No prospects to import.' });
            return;
        }

        if (sourceType === 'existing') {
            if (!existingListId) {
                toast({ variant: 'destructive', title: 'No List Selected', description: 'Please select an audience list.' });
                return;
            }
            const listToUpdate = audienceLists.find(l => l.id === existingListId);
            if (listToUpdate) {
                const updatedProspects = [...(listToUpdate.prospects || []), ...prospects];
                 await updateAudienceList(existingListId, { prospects: updatedProspects, count: updatedProspects.length });
                toast({ title: 'Import Successful', description: `${prospects.length} prospects added to "${listToUpdate.name}".` });

            }
        } else { // New List
            if (!newListName) {
                toast({ variant: 'destructive', title: 'Missing List Name', description: 'Please provide a name for the new list.' });
                return;
            }
             await addAudienceList({
                 name: newListName,
                 status: 'Saved',
                 source: 'Import',
                 count: prospects.length,
                 prospects: prospects
             }, currentUserId);
            toast({ title: 'Import Successful', description: `${prospects.length} prospects added to new list "${newListName}".` });
        }
        resetState();
        setOpen(false);
        router.refresh()
    };

    return (
        <Dialog open={open} onOpenChange={(isOpen) => {
            setOpen(isOpen);
            if (!isOpen) resetState();
        }}>
            <DialogTrigger asChild>
                <Button variant="outline">
                    <Upload className="mr-2 h-4 w-4" />
                    Import List
                </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl">
                <DialogHeader>
                    <DialogTitle>Import Prospects from CSV</DialogTitle>
                    <DialogDescription>Upload a CSV file to create a new audience list or add to an existing one.</DialogDescription>
                </DialogHeader>
                <div className="grid gap-6 py-4">
                    <div {...getRootProps()} className={cn("flex items-center justify-center w-full h-48 border-2 border-dashed rounded-lg cursor-pointer bg-muted hover:bg-muted/80", isDragActive && "border-primary")}>
                        <input {...getInputProps()} />
                        {uploadedFile ? (
                             <div className="text-center">
                                <Download className="w-12 h-12 mx-auto mb-4 text-green-600" />
                                <p className="font-semibold">{uploadedFile.name}</p>
                                <p className="text-sm text-muted-foreground">{prospects.length} prospects ready</p>
                                <Button type="button" variant="link" size="sm" onClick={(e) => { e.stopPropagation(); resetState(); }}>Remove file</Button>
                            </div>
                        ) : (
                            <button type="button" onClick={openFileDialog} className="flex flex-col items-center justify-center pt-5 pb-6">
                                <Upload className="w-8 h-8 mb-4 text-muted-foreground" />
                                <p className="mb-2 text-sm text-muted-foreground"><span className="font-semibold">Click to upload</span> or drag and drop</p>
                                <p className="text-xs text-muted-foreground">CSV with 'Full Name' and 'Email' headers</p>
                            </button>
                        )}
                    </div>

                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label className="text-right">Assign to</Label>
                        <RadioGroup value={sourceType} onValueChange={(v) => setSourceType(v as any)} className="col-span-3 flex gap-4" disabled={!uploadedFile}>
                             <div className="flex items-center space-x-2">
                                <RadioGroupItem value="existing" id="import-r-existing" />
                                <Label htmlFor="import-r-existing">Existing list</Label>
                            </div>
                            <div className="flex items-center space-x-2">
                                <RadioGroupItem value="new" id="import-r-new" />
                                <Label htmlFor="import-r-new">New list</Label>
                            </div>
                        </RadioGroup>
                    </div>

                     {sourceType === 'new' && (
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="import-new-list-name" className="text-right">New List Name</Label>
                            <Input id="import-new-list-name" value={newListName} onChange={(e) => setNewListName(e.target.value)} className="col-span-3" placeholder="e.g., Imported Q2 Leads"/>
                        </div>
                    )}
                    
                    {sourceType === 'existing' && (
                        <div className="grid grid-cols-4 items-center gap-4">
                             <Label htmlFor="import-existing-list" className="text-right">Audience List</Label>
                             <Select onValueChange={setExistingListId} value={existingListId} disabled={!uploadedFile}>
                                <SelectTrigger className="col-span-3">
                                    <SelectValue placeholder="Select a list..." />
                                </SelectTrigger>
                                <SelectContent>
                                    {audienceLists.filter(l => l.status === 'Saved' && l.source === 'Import').map(list => (
                                        <SelectItem key={list.id} value={list.id}>{list.name}</SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                    )}

                </div>
                <DialogFooter>
                    <Button type="button" variant="outline" onClick={() => setOpen(false)}>Cancel</Button>
                    <Button type="button" onClick={handleImport} disabled={prospects.length === 0}>Import Prospects</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

export default function ProspectsPage() {
    const { toast } = useToast();
    const { audienceLists, deleteProspect, updateProspect } = useAudience();
    const [searchTerm, setSearchTerm] = useState('');
    const [activeTab, setActiveTab] = useState('all');

    const [selectedProspect, setSelectedProspect] = useState<ProspectSource | null>(null);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isDeleteOpen, setIsDeleteOpen] = useState(false);
    const [sortConfig, setSortConfig] = useState<SortConfig>({ key: 'lastModified', direction: 'desc' });

    const allProspects = useMemo(() => {
        return audienceLists.flatMap(list => 
            (list.prospects || []).map(prospect => ({
                ...prospect,
                source: list.name,
                sourceType: list.source,
                sourceId: list.id,
                lastModified: list.last_modified
            }))
        );
    }, [audienceLists]);
    
    const prospectsToShow = useMemo(() => {
        let listsToProcess: AudienceList[] = [];
        if (activeTab === 'all') {
            listsToProcess = audienceLists;
        } else if (activeTab === 'audience') {
            listsToProcess = audienceLists.filter(l => l.source === 'Import');
        } else if (activeTab === 'one-off') {
            listsToProcess = audienceLists.filter(l => l.source === 'One-Off');
        }

        return listsToProcess.flatMap(list => 
            (list.prospects || []).map(prospect => ({
                ...prospect,
                source: list.name,
                sourceType: list.source,
                sourceId: list.id,
                lastModified: list.last_modified
            }))
        );

    }, [audienceLists, activeTab]);


    const filteredProspects = useMemo(() => {
        let searched = prospectsToShow.filter(p =>
            p.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
            p.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
            p.source.toLowerCase().includes(searchTerm.toLowerCase())
        );

        if (sortConfig) {
            searched.sort((a, b) => {
                if (sortConfig.key === 'lastModified') {
                    const dateA = new Date((a[sortConfig.key] as any) || 0).getTime();
                    const dateB = new Date((b[sortConfig.key] as any) || 0).getTime();
                    return sortConfig.direction === 'asc' ? dateA - dateB : dateB - dateA;
                } else {
                     const aValue = a[sortConfig.key as keyof ProspectSource] as any;
                     const bValue = b[sortConfig.key as keyof ProspectSource] as any;
                    if (aValue < bValue) return sortConfig.direction === 'asc' ? -1 : 1;
                    if (aValue > bValue) return sortConfig.direction === 'asc' ? 1 : -1;
                    return 0;
                }
            });
        }
        return searched;
    }, [prospectsToShow, searchTerm, sortConfig]);
    
    const handleExport = () => {
        const prospectsToExport = filteredProspects;
        let filename = 'all_prospects.csv';
        if (activeTab === 'audience') {
            filename = 'audience_prospects.csv';
        } else if (activeTab === 'one-off') {
            filename = 'one_off_prospects.csv';
        }


        if (prospectsToExport.length === 0) {
            toast({ variant: 'destructive', title: 'No Prospects to Export' });
            return;
        }
        const headers = ["Full Name", "Email"];
        const csvRows = [
            headers.join(','),
            ...prospectsToExport.map(p => [
                `"${p.name.replace(/"/g, '""')}"`,
                `"${p.email.replace(/"/g, '""')}"`
            ].join(','))
        ];
        
        const blob = new Blob([csvRows.join('\n')], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        link.setAttribute('href', url);
        link.setAttribute('download', filename);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        toast({ title: 'Export Successful', description: `Your prospects have been downloaded as ${filename}` });
    }

    const handleDeleteClick = (prospect: ProspectSource) => {
        setSelectedProspect(prospect);
        setIsDeleteOpen(true);
    };
    
    const handleEditClick = (prospect: ProspectSource) => {
        setSelectedProspect(prospect);
        setIsEditOpen(true);
    };
    
    const handleDeleteConfirm = async () => {
        if (selectedProspect) {
            await deleteProspect(selectedProspect.id, selectedProspect.sourceId);
            toast({ variant: 'destructive', title: 'Prospect Deleted', description: `${selectedProspect.name} has been removed.` });
            setIsDeleteOpen(false);
            setSelectedProspect(null);
        }
    };
    
    const handleSort = (key: keyof ProspectSource | 'lastModified', direction: 'asc' | 'desc') => {
      setSortConfig({ key, direction });
    }

    const sortLabels: Record<string, string> = {
        'lastModified-desc': 'Latest',
        'lastModified-asc': 'Oldest',
        'name-asc': 'Name (A-Z)',
        'name-desc': 'Name (Z-A)',
    }
    const currentSortLabel = sortConfig ? sortLabels[`${sortConfig.key}-${sortConfig.direction}`] : 'Sort by';

    return (
        <>
            <PageHeader
                title="Prospects"
                description="A unified list of all your contacts."
            >
                <div className="flex gap-2">
                    <AddProspectDialog />
                    <ImportProspectsDialog />
                    <Button onClick={handleExport} variant="outline">
                        <Download className="mr-2 h-4 w-4" />
                        Export
                    </Button>
                </div>
            </PageHeader>

            <Tabs defaultValue="all" onValueChange={setActiveTab} className="space-y-4">
                <div className="flex justify-between items-center">
                    <TabsList>
                        <TabsTrigger value="all">All Contacts ({allProspects.length})</TabsTrigger>
                        <TabsTrigger value="audience">From Audience Lists ({allProspects.filter(p => p.sourceType === 'Import').length})</TabsTrigger>
                        <TabsTrigger value="one-off">One-Offs ({allProspects.filter(p => p.sourceType === 'One-Off').length})</TabsTrigger>
                    </TabsList>
                </div>
                
                <Card>
                     <CardHeader>
                        <div className="flex justify-between items-center mb-4">
                             <div>
                                <CardTitle>Prospect List</CardTitle>
                                <CardDescription>
                                    {filteredProspects.length} prospect{filteredProspects.length === 1 ? '' : 's'} found.
                                </CardDescription>
                            </div>
                        </div>
                        <div className="flex gap-2">
                             <div className="relative flex-1">
                                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                <Input
                                    placeholder="Search contacts..."
                                    className="pl-10"
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                <Button variant="outline">{currentSortLabel} <ChevronDown className="ml-2 h-4 w-4" /></Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent>
                                <DropdownMenuItem onClick={() => handleSort('lastModified', 'desc')}>
                                    <Clock className="mr-2 h-4 w-4" /> Last Modified (Latest)
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('lastModified', 'asc')}>
                                    <Clock className="mr-2 h-4 w-4" /> Last Modified (Oldest)
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('name', 'asc')}>
                                    <ArrowDownAZ className="mr-2 h-4 w-4" /> Name (A-Z)
                                </DropdownMenuItem>
                                <DropdownMenuItem onClick={() => handleSort('name', 'desc')}>
                                    <ArrowUpAZ className="mr-2 h-4 w-4" /> Name (Z-A)
                                </DropdownMenuItem>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </CardHeader>
                    <CardContent>
                        <ProspectsTable prospects={filteredProspects} onEdit={handleEditClick} onDelete={handleDeleteClick} />
                    </CardContent>
                </Card>
            </Tabs>

            <EditProspectDialog prospect={selectedProspect} open={isEditOpen} onOpenChange={setIsEditOpen} onUpdate={updateProspect} />
            
            <AlertDialog open={isDeleteOpen} onOpenChange={setIsDeleteOpen}>
                <AlertDialogContent>
                    <AlertDialogHeader>
                        <AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle>
                        <AlertDialogDescription>
                            This will permanently delete the prospect "{selectedProspect?.name}" from the list "{selectedProspect?.source}". This action cannot be undone.
                        </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter>
                        <AlertDialogCancel onClick={() => setSelectedProspect(null)}>Cancel</AlertDialogCancel>
                        <AlertDialogAction onClick={handleDeleteConfirm}>Delete</AlertDialogAction>
                    </AlertDialogFooter>
                </AlertDialogContent>
            </AlertDialog>
        </>
    );
}


function ProspectsTable({ prospects, onEdit, onDelete }: { prospects: ProspectSource[], onEdit: (prospect: ProspectSource) => void, onDelete: (prospect: ProspectSource) => void }) {
    return (
        <Table>
            <TableHeader>
                <TableRow>
                    <TableHead>Name</TableHead>
                    <TableHead>Email</TableHead>
                    <TableHead>Source List</TableHead>
                    <TableHead>Last Modified</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                </TableRow>
            </TableHeader>
            <TableBody>
                {prospects.length > 0 ? (
                    prospects.map((prospect) => {
                        const lastModified = prospect.lastModified ? format(new Date(prospect.lastModified), 'PPP') : 'N/A';
                        const isOneOff = (prospect as any).sourceType === 'One-Off';
                        
                        return (
                            <TableRow key={`${prospect.email}-${prospect.sourceId}`}>
                                <TableCell className="font-medium">{prospect.name}</TableCell>
                                <TableCell>{prospect.email}</TableCell>
                                <TableCell>
                                    <Badge variant={isOneOff ? 'secondary' : 'outline'}>{isOneOff ? 'One-Off' : prospect.source}</Badge>
                                </TableCell>
                                <TableCell>{lastModified}</TableCell>
                                <TableCell className="text-right">
                                    <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button variant="ghost" className="h-8 w-8 p-0">
                                                <span className="sr-only">Open menu</span>
                                                <MoreHorizontal className="h-4 w-4" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent align="end">
                                            <DropdownMenuItem onClick={() => onEdit(prospect)}>
                                                <Edit className="mr-2 h-4 w-4" />
                                                Edit Prospect
                                            </DropdownMenuItem>
                                            <DropdownMenuSeparator />
                                            <DropdownMenuItem onClick={() => onDelete(prospect)} className="text-destructive">
                                                <Trash2 className="mr-2 h-4 w-4" />
                                                Delete Prospect
                                            </DropdownMenuItem>
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                            </TableCell>
                        </TableRow>
                    )
                })
                ) : (
                    <TableRow>
                        <TableCell colSpan={5} className="h-24 text-center">
                            No prospects found.
                        </TableCell>
                    </TableRow>
                )}
            </TableBody>
        </Table>
    )
}
