'use client';

import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Send, UserPlus, X, FileText, CheckCircle, Loader2, ChevronsUpDown, Globe } from "lucide-react";
import { useAudience } from "@/context/AudienceContext";
import { useState, useMemo, useEffect, Suspense, useTransition } from "react";
import type { Prospect, Template } from "@/lib/types";
import { useToast } from "@/hooks/use-toast";
import { useTemplates } from "@/context/TemplateContext";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Checkbox } from "@/components/ui/checkbox";


// --- NEW IMPORTS ---
import { getAvailableSenders } from '../campaigns/actions';
import { getVerifiedTrackingDomains } from '../settings/domains/actions';
import { sendOneOffEmail } from './actions';

type Sender = { id: number; name: string; service_type: string };
type VerifiedDomain = { id: number; domain_name: string };

function AddProspectDialog({ onAddProspect }: { onAddProspect: (prospect: Prospect) => void }) {
    const [open, setOpen] = useState(false);
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const { toast } = useToast();

    const handleSubmit = () => {
        if (!name || !email) {
            toast({ variant: 'destructive', title: 'Missing fields' });
            return;
        }
        const newProspect: Prospect = { id: `MANUAL-${Date.now()}`, name, email };
        onAddProspect(newProspect);
        setName(''); setEmail(''); setOpen(false);
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild><Button variant="outline" size="icon"><UserPlus /></Button></DialogTrigger>
            <DialogContent>
                <DialogHeader><DialogTitle>Add New Prospect</DialogTitle><DialogDescription>Manually add a new prospect for a one-off email.</DialogDescription></DialogHeader>
                <div className="grid gap-4 py-4">
                    <div className="grid grid-cols-4 items-center gap-4"><Label htmlFor="prospect-name" className="text-right">Name</Label><Input id="prospect-name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" /></div>
                    <div className="grid grid-cols-4 items-center gap-4"><Label htmlFor="prospect-email" className="text-right">Email</Label><Input id="prospect-email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" /></div>
                </div>
                <DialogFooter><Button type="button" onClick={handleSubmit}>Add Prospect</Button></DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function SelectTemplateDialog({ onSelectTemplate, open, onOpenChange }: { onSelectTemplate: (template: Template) => void, open: boolean, onOpenChange: (open: boolean) => void }) {
    const { templates } = useTemplates();
    const handleSelect = (template: Template) => { onSelectTemplate(template); onOpenChange(false); };
    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-2xl">
                <DialogHeader><DialogTitle>Select an Email Template</DialogTitle><DialogDescription>Choose from your saved templates to send.</DialogDescription></DialogHeader>
                <div className="py-4 max-h-[60vh] overflow-y-auto">
                    <div className="space-y-2">
                        {templates.map(template => ( <button key={template.id} onClick={() => handleSelect(template)} className="w-full text-left p-3 border rounded-lg hover:bg-muted/50"><p className="font-semibold">{template.name}</p><p className="text-sm text-muted-foreground truncate">Subject: {template.subject}</p></button>))}
                        {templates.length === 0 && ( <p className="text-center text-muted-foreground py-8">No templates found.</p> )}
                    </div>
                </div>
            </DialogContent>
        </Dialog>
    );
}

export default function OneOffEmailPage() {
    return ( <Suspense fallback={<div>Loading...</div>}><OneOffEmailContent /></Suspense> );
}

function OneOffEmailContent() {
    const { audienceLists } = useAudience();
    const { toast } = useToast();
    const [isPending, startTransition] = useTransition();

    // --- UPGRADED STATE ---
    const [selectedProspect, setSelectedProspect] = useState<Prospect | null>(null);
    const [selectedTemplate, setSelectedTemplate] = useState<Template | null>(null);
    const [availableSenders, setAvailableSenders] = useState<Sender[]>([]);
    const [verifiedDomains, setVerifiedDomains] = useState<VerifiedDomain[]>([]);
    const [selectedSenderId, setSelectedSenderId] = useState<number | null>(null);
    const [selectedTrackingDomainId, setSelectedTrackingDomainId] = useState<number | null>(null);
    const [warmup, setWarmup] = useState(false);
    const [throttling, setThrottling] = useState(false);

    const [openProspectCombobox, setOpenProspectCombobox] = useState(false);
    const [isTemplateDialogOpen, setIsTemplateDialogOpen] = useState(false);
    const [searchValue, setSearchValue] = useState("");

    useEffect(() => {
        getAvailableSenders().then(setAvailableSenders);
        getVerifiedTrackingDomains().then(setVerifiedDomains); // <-- Fetch domains on load
    }, []);

    const allProspects = useMemo(() => {
        const prospectMap = new Map<string, Prospect>();
        audienceLists.forEach(list => { (list.prospects || []).forEach(prospect => { if (prospect?.email && !prospectMap.has(prospect.email)) prospectMap.set(prospect.email, prospect); }) });
        return Array.from(prospectMap.values());
    }, [audienceLists]);

    const handleAddProspect = (prospect: Prospect) => {
        setSelectedProspect(prospect);
        toast({ title: 'Prospect Selected', description: `${prospect.name} has been set as the recipient.` });
    };

    const handleSendEmail = async () => {
        if (!selectedProspect || !selectedTemplate || !selectedSenderId || !selectedTrackingDomainId) {
            toast({ variant: 'destructive', title: 'Missing Information', description: 'Please select a recipient, template, sender, and tracking domain.' });
            return;
        }

        startTransition(async () => {
            const result = await sendOneOffEmail({
                prospect: selectedProspect,
                templateId: selectedTemplate.id,
                senderId: selectedSenderId,
                trackingDomainId: selectedTrackingDomainId, // <-- PASS THE ID
                warmup,
                throttling
            });

            if (result.success) {
                toast({ title: 'Email Scheduled!', description: result.message });
                setSelectedProspect(null);
                setSelectedTemplate(null);
                setSelectedSenderId(null);
                setSelectedTrackingDomainId(null);
            } else {
                 toast({ variant: 'destructive', title: 'Send Failed', description: result.message });
            }
        });
    };
    
    return (
        <>
            <PageHeader
                title="One-Off Email Sender"
                description="Send single, highly personalized emails using your templates and senders."
            />
            <div className="grid gap-6">
                <Card>
                    <CardHeader><CardTitle>1. Select Recipient</CardTitle><CardDescription>Search for a prospect from your audience lists or add a new one manually.</CardDescription></CardHeader>
                    <CardContent className="space-y-4">
                       {selectedProspect ? (
                             <div className="flex items-center gap-2 p-2 rounded-md border bg-muted/50 flex-1"><p className="font-semibold text-sm">{selectedProspect.name}</p><p className="text-sm text-muted-foreground">{selectedProspect.email}</p><Button variant="ghost" size="icon" onClick={() => setSelectedProspect(null)} className="ml-auto h-6 w-6"><X className="h-4 w-4" /></Button></div>
                       ) : (
                        <div className="flex items-center gap-2">
                            <Popover open={openProspectCombobox} onOpenChange={setOpenProspectCombobox}>
                                <PopoverTrigger asChild><Button variant="outline" role="combobox" className="w-[300px] justify-between">{"Select from prospects..."}<ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" /></Button></PopoverTrigger>
                                <PopoverContent className="w-[300px] p-0"><Command><CommandInput placeholder="Search prospect..." value={searchValue} onValueChange={setSearchValue}/><CommandList><CommandEmpty>No prospect found.</CommandEmpty><CommandGroup>
                                            {allProspects.filter(p => p.name.toLowerCase().includes(searchValue.toLowerCase()) || p.email.toLowerCase().includes(searchValue.toLowerCase())).map(p => (
                                                <CommandItem key={p.id} onSelect={() => { setSelectedProspect(p); setOpenProspectCombobox(false); setSearchValue(""); }}>
                                                    <div><p>{p.name}</p><p className="text-xs text-muted-foreground">{p.email}</p></div>
                                                </CommandItem>
                                            ))}
                                </CommandGroup></CommandList></Command></PopoverContent>
                            </Popover>
                             <AddProspectDialog onAddProspect={handleAddProspect} />
                        </div>
                       )}
                    </CardContent>
                </Card>

                <Card>
                    <CardHeader><CardTitle>2. Select Template</CardTitle><CardDescription>Choose the email template you want to send.</CardDescription></CardHeader>
                    <CardContent>
                       {!selectedTemplate ? ( <Button variant="outline" onClick={() => setIsTemplateDialogOpen(true)}><FileText className="mr-2 h-4 w-4"/>Choose a Template</Button> ) : (
                           <div className="flex items-center gap-2 p-3 rounded-md border bg-muted/50"><CheckCircle className="h-5 w-5 text-green-600" /><div className="flex-1"><p className="font-semibold text-sm">{selectedTemplate.name}</p><p className="text-xs text-muted-foreground truncate">Subject: {selectedTemplate.subject}</p></div><Button variant="ghost" size="icon" onClick={() => setSelectedTemplate(null)} className="ml-auto h-6 w-6"><X className="h-4 w-4" /></Button></div>
                       )}
                    </CardContent>
                </Card>

                <Card>
                     <CardHeader><CardTitle>3. Configure & Send</CardTitle><CardDescription>Choose your sending options and send the email.</CardDescription></CardHeader>
                    <CardContent className="space-y-6">
                        {/* --- UPGRADED SENDER SELECTION --- */}
                        <div className="grid gap-2">
                            <Label className="font-semibold">Sending Account</Label>
                            <Select onValueChange={(value) => setSelectedSenderId(Number(value))} disabled={availableSenders.length === 0}>
                                <SelectTrigger><SelectValue placeholder="Select a sender..." /></SelectTrigger>
                                <SelectContent>
                                    {availableSenders.map(sender => (
                                        <SelectItem key={sender.id} value={sender.id.toString()}>
                                            {sender.name} ({sender.service_type})
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        {/* --- THIS IS THE NEW DROPDOWN --- */}
                        <div className="grid gap-2">
                                <Label className="font-semibold flex items-center gap-2"><Globe className="h-4 w-4" /> Tracking Domain</Label>
                                <Select onValueChange={(value) => setSelectedTrackingDomainId(Number(value))} disabled={verifiedDomains.length === 0}>
                                    <SelectTrigger><SelectValue placeholder="Select a tracking domain..." /></SelectTrigger>
                                    <SelectContent>
                                        {verifiedDomains.map(domain => (<SelectItem key={domain.id} value={domain.id.toString()}>{domain.domain_name}</SelectItem>))}
                                    </SelectContent>
                                </Select>
                        </div>

                        <div className="grid gap-4">
                            <Label className="font-semibold">Deliverability Optimization</Label>
                            <div className="flex items-center space-x-2"><Checkbox id="warmup" checked={warmup} onCheckedChange={(c) => setWarmup(!!c)} /><Label htmlFor="warmup">Enable Warm-up Automation</Label></div>
                            <div className="flex items-center space-x-2"><Checkbox id="throttling" checked={throttling} onCheckedChange={(c) => setThrottling(!!c)} /><Label htmlFor="throttling">Enable Throttling</Label></div>
                        </div>

                        <Button onClick={handleSendEmail} className="w-full" disabled={!selectedProspect || !selectedTemplate || !selectedSenderId || isPending}>
                           {isPending ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Send className="mr-2 h-4 w-4" />}
                           {isPending ? 'Scheduling...' : 'Schedule & Send Email'}
                        </Button>
                    </CardContent>
                </Card>
            </div>
            <SelectTemplateDialog 
                open={isTemplateDialogOpen}
                onOpenChange={setIsTemplateDialogOpen}
                onSelectTemplate={setSelectedTemplate}
            />
        </>
    );
}