
import { PageHeader } from "@/components/page-header";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
    Card,
    CardContent,
    CardDescription,
    CardHeader,
    CardTitle 
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Progress } from "@/components/ui/progress";
import { Separator } from "@/components/ui/separator";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { mockCampaigns, mockOneOffAnalytics } from "@/lib/mock-data";
import { ShieldCheck, ShieldOff, MailWarning, ListX, TestTube, Dna, Activity, Search } from "lucide-react";

export default function DeliveryHealthPage() {
    return (
        <>
            <PageHeader
                title="Compliance & Delivery Health"
                description="Monitor and improve your sender reputation and deliverability."
            />
            
            {/* Health Check Section */}
            <Card className="mb-8">
                <CardHeader>
                    <CardTitle>Individual Health Check</CardTitle>
                    <CardDescription>Search for a specific campaign or one-off email to run a detailed health check.</CardDescription>
                </CardHeader>
                <CardContent>
                    <div className="flex gap-2 mb-4">
                        <div className="relative flex-1">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                            <Input placeholder="Search by campaign name or recipient..." className="pl-10" />
                        </div>
                        <Button>Run Health Check</Button>
                    </div>
                     <Tabs defaultValue="campaigns">
                        <TabsList>
                            <TabsTrigger value="campaigns">Campaigns</TabsTrigger>
                            <TabsTrigger value="one-offs">One-Offs</TabsTrigger>
                        </TabsList>
                        <TabsContent value="campaigns">
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>Campaign Name</TableHead>
                                        <TableHead>Status</TableHead>
                                        <TableHead>Spam Score</TableHead>
                                        <TableHead>Health Status</TableHead>
                                        <TableHead className="text-right">Actions</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {mockCampaigns.slice(0, 3).map((campaign) => (
                                        <TableRow key={campaign.id}>
                                            <TableCell className="font-medium">{campaign.name}</TableCell>
                                            <TableCell><Badge variant={campaign.status === 'Sent' ? 'default' : 'secondary'}>{campaign.status}</Badge></TableCell>
                                            <TableCell>Low</TableCell>
                                            <TableCell><Badge className="bg-green-600">Healthy</Badge></TableCell>
                                            <TableCell className="text-right">
                                                <Button variant="outline" size="sm">View Report</Button>
                                            </TableCell>
                                        </TableRow>
                                    ))}
                                </TableBody>
                            </Table>
                        </TabsContent>
                        <TabsContent value="one-offs">
                             <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>Recipient</TableHead>
                                        <TableHead>Subject</TableHead>
                                        <TableHead>Spam Score</TableHead>
                                        <TableHead>Status</TableHead>
                                        <TableHead className="text-right">Actions</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {mockOneOffAnalytics.slice(0, 3).map((email) => (
                                        <TableRow key={email.id}>
                                            <TableCell className="font-medium">{email.recipient}</TableCell>
                                            <TableCell>{email.subject}</TableCell>
                                            <TableCell>Low</TableCell>
                                            <TableCell><Badge className="bg-green-600">Delivered</Badge></TableCell>
                                            <TableCell className="text-right">
                                                 <Button variant="outline" size="sm">View Report</Button>
                                            </TableCell>
                                        </TableRow>
                                    ))}
                                </TableBody>
                            </Table>
                        </TabsContent>
                    </Tabs>
                </CardContent>
            </Card>

            <div className="grid gap-8 lg:grid-cols-3">
                {/* Left Column */}
                <div className="lg:col-span-2 space-y-8">
                    <Card>
                        <CardHeader>
                            <CardTitle>Overall Domain Health</CardTitle>
                            <CardDescription>Overview of your sending domain's reputation.</CardDescription>
                        </CardHeader>
                        <CardContent className="grid gap-6">
                             <div className="flex items-center">
                                <span className="w-32 font-medium">Domain Reputation</span>
                                <Progress value={92} className="flex-1" />
                                <span className="ml-4 text-lg font-bold">92%</span>
                            </div>
                             <div className="flex items-center">
                                <span className="w-32 font-medium">Spam Trap Hits</span>
                                 <Progress value={5} className="flex-1 [&>*]:bg-destructive" />
                                <span className="ml-4 text-lg font-bold">5%</span>
                            </div>
                            <Separator />
                            <div className="grid md:grid-cols-3 gap-4 text-center">
                                <div>
                                    <p className="text-sm text-muted-foreground">DKIM</p>
                                    <Badge variant="default" className="bg-green-600">Aligned</Badge>
                                </div>
                                 <div>
                                    <p className="text-sm text-muted-foreground">SPF</p>
                                    <Badge variant="default" className="bg-green-600">Aligned</Badge>
                                </div>
                                 <div>
                                    <p className="text-sm text-muted-foreground">DMARC</p>
                                    <Badge variant="destructive">Missing</Badge>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader>
                             <CardTitle>Active Monitoring</CardTitle>
                            <CardDescription>Real-time checks and balances for your campaigns.</CardDescription>
                        </CardHeader>
                        <CardContent className="grid md:grid-cols-2 gap-4">
                            <div className="flex items-center gap-4 p-4 border rounded-lg">
                                <Dna className="h-8 w-8 text-primary" />
                                <div>
                                    <p className="font-semibold">Reply/Bounce Detection</p>
                                    <p className="text-sm text-muted-foreground">Auto-enabled</p>
                                </div>
                            </div>
                             <div className="flex items-center gap-4 p-4 border rounded-lg">
                                <ListX className="h-8 w-8 text-primary" />
                                <div>
                                    <p className="font-semibold">Suppression List Sync</p>
                                    <p className="text-sm text-muted-foreground">Syncing daily</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>
                {/* Right Column */}
                <div className="space-y-8">
                     <Card>
                        <CardHeader>
                            <CardTitle>Pre-Send Checks</CardTitle>
                            <CardDescription>Tools to ensure your emails land in the inbox.</CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-4">
                           <div className="flex items-start gap-4 p-4 border rounded-lg bg-muted/30">
                               <MailWarning className="h-6 w-6 text-amber-500 mt-1" />
                               <div>
                                   <p className="font-semibold">Spam Score Warnings</p>
                                   <p className="text-sm text-muted-foreground mb-2">Check your email content against spam filters before sending.</p>
                                   <Button variant="outline">Run Spam Check</Button>
                               </div>
                           </div>
                           <div className="flex items-start gap-4 p-4 border rounded-lg bg-muted/30">
                               <TestTube className="h-6 w-6 text-blue-500 mt-1" />
                               <div>
                                   <p className="font-semibold">Seed/Test Sending</p>
                                   <p className="text-sm text-muted-foreground mb-2">Send a test to a list of seed emails to check inbox placement.</p>
                                   <Button variant="outline">Send Seed Test</Button>
                               </div>
                           </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader>
                            <CardTitle>Compliance</CardTitle>
                             <CardDescription>Automated compliance features.</CardDescription>
                        </CardHeader>
                        <CardContent>
                             <div className="flex items-center gap-4">
                                <ShieldCheck className="h-8 w-8 text-green-600" />
                                <div>
                                    <p className="font-semibold">Auto-Unsubscribe Links</p>
                                    <p className="text-sm text-muted-foreground">Enabled on all outgoing emails.</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </>
    );
}
