'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import type { Campaign } from '@/lib/types';
import { getUserId } from '@/lib/auth';

/**
 * Creates a new campaign and its associated senders in a single transaction.
 */
export async function createCampaign(campaignData: Omit<Campaign, 'id' | 'analytics' | 'user_id' | 'audience'>) {
  const { name, objective, status, audienceListId, sequence, senderIds, trackingDomainId, sender } = campaignData;
  const userId = await getUserId();
  if (!userId) return { success: false, message: "Authentication failed." };

  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    const campaignInsertQuery = `
      INSERT INTO campaigns (user_id, name, objective, status, audience_list_id, sequence, warmup, throttling, tracking_domain_id)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
      RETURNING id;
    `;
    const campaignResult = await client.query(campaignInsertQuery, [
      userId, name, objective, status, audienceListId, JSON.stringify(sequence),
      sender.warmup, sender.throttling, trackingDomainId
    ]);
    const newCampaignId = campaignResult.rows[0].id;

    if (senderIds && senderIds.length > 0) {
      for (const serviceId of senderIds) {
        await client.query(
          'INSERT INTO campaign_senders (campaign_id, smtp_service_id) VALUES ($1, $2)',
          [newCampaignId, serviceId]
        );
      }
    }

    await client.query('COMMIT');
    revalidatePath('/campaigns');
    return { success: true, message: 'Campaign created successfully!' };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error("Database Error: Failed to create campaign.", error);
    return { success: false, message: 'A database error occurred.' };
  } finally {
    client.release();
  }
}

/**
 * --- THIS IS THE KEY BACKEND FIX ---
 * Fetches all campaigns for a user and perfectly maps the data to the expected Campaign type.
 */
export async function getCampaigns(): Promise<Campaign[]> {
    const userId = await getUserId();
    if (!userId) return [];
    try {
        const result = await pool.query(`
            SELECT
                c.id, c.user_id, c.name, c.status, c.objective, c.sequence,
                c.warmup, c.throttling, c.last_modified, c.created_at,
                c.audience_list_id, c.tracking_domain_id,
                a.name as audience_name,
                a.count as audience_count,
                (
                    SELECT array_agg(cs.smtp_service_id)
                    FROM campaign_senders cs
                    WHERE cs.campaign_id = c.id
                ) as sender_ids
            FROM campaigns c
            LEFT JOIN audiences a ON c.audience_list_id = a.id
            WHERE c.user_id = $1
            ORDER BY c.last_modified DESC
        `, [userId]);

        // This map function is the "perfect translator"
        return result.rows.map(row => ({
            id: row.id,
            user_id: row.user_id,
            name: row.name,
            status: row.status,
            objective: row.objective,
            sequence: row.sequence,
            audienceListId: row.audience_list_id,
            trackingDomainId: row.tracking_domain_id,
            // It consistently translates all snake_case fields to camelCase
            createdAt: row.created_at,
            lastModified: row.last_modified,
            audience: {
                id: row.audience_list_id,
                name: row.audience_name,
                count: row.audience_count || 0,
            },
            sender: {
                warmup: row.warmup,
                throttling: row.throttling,
            },
            senderIds: row.sender_ids || [],
            analytics: { sent: 0, openRate: 0, clickRate: 0 },
        })) as Campaign[];
    } catch (error) {
        console.error("Database Error: Failed to fetch campaigns.", error);
        return [];
    }
}

/**
 * Fetches all available sender services.
 */
export async function getAvailableSenders() {
  const userId = await getUserId();
  if (!userId) return [];
  try {
    const res = await pool.query("SELECT id, name, service_type FROM smtp_services ORDER BY name");
    return res.rows;
  } catch (error) {
    console.error("Database Error: Failed to fetch senders.", error);
    return [];
  }
}

