'use client';

import { useState, useEffect, useMemo } from 'react';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import { Label } from '@/components/ui/label';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Separator } from '@/components/ui/separator';

interface Sender {
  id: number;
  name: string;
  service_type: string; // This property is required for grouping
}

interface SenderSelectionDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  availableSenders: Sender[];
  selectedSenderIds: number[];
  onSave: (selectedIds: number[]) => void;
}

// Helper function to format the group titles
const formatGroupTitle = (type: string) => {
    switch (type) {
        case 'gmail': return 'Gmail / Google Workspace';
        case 'custom': return 'Custom SMTP';
        case 'platform': return 'Platform Senders';
        default: return 'Other';
    }
};

export function SenderSelectionDialog({
  open,
  onOpenChange,
  availableSenders,
  selectedSenderIds,
  onSave,
}: SenderSelectionDialogProps) {
  const [currentSelections, setCurrentSelections] = useState<Set<number>>(new Set());

  // Group senders by their service_type
  const groupedSenders = useMemo(() => {
    return availableSenders.reduce((acc, sender) => {
      const type = sender.service_type || 'other';
      if (!acc[type]) {
        acc[type] = [];
      }
      acc[type].push(sender);
      return acc;
    }, {} as Record<string, Sender[]>);
  }, [availableSenders]);

  useEffect(() => {
    if (open) {
      setCurrentSelections(new Set(selectedSenderIds));
    }
  }, [open, selectedSenderIds]);

  const handleCheckboxChange = (senderId: number, checked: boolean) => {
    setCurrentSelections(prev => {
      const newSelections = new Set(prev);
      if (checked) {
        newSelections.add(senderId);
      } else {
        newSelections.delete(senderId);
      }
      return newSelections;
    });
  };

  const handleSave = () => {
    onSave(Array.from(currentSelections));
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Select Senders for Rotation</DialogTitle>
          <DialogDescription>
            Choose one or more sending services. The system will rotate through them to send your emails.
          </DialogDescription>
        </DialogHeader>

        <ScrollArea className="h-80 w-full pr-4">
            <div className="space-y-6">
            {Object.entries(groupedSenders).length > 0 ? (
                Object.entries(groupedSenders).map(([type, senders]) => (
                    <div key={type} className="space-y-3">
                        <h4 className="text-sm font-semibold text-muted-foreground">{formatGroupTitle(type)}</h4>
                        {senders.map((sender) => (
                            <div key={sender.id} className="flex items-center space-x-3 rounded-md border p-3 transition-colors hover:bg-muted/50">
                                <Checkbox
                                id={`sender-${sender.id}`}
                                checked={currentSelections.has(sender.id)}
                                onCheckedChange={(checked) => handleCheckboxChange(sender.id, !!checked)}
                                />
                                <Label htmlFor={`sender-${sender.id}`} className="font-medium flex-1 cursor-pointer">
                                {sender.name}
                                </Label>
                            </div>
                        ))}
                    </div>
                ))
            ) : (
                <p className="py-10 text-center text-sm text-muted-foreground">No verified senders found.</p>
            )}
            </div>
        </ScrollArea>

        <DialogFooter>
          <Button type="button" onClick={handleSave}>
            Save Selections ({currentSelections.size} selected)
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
