'use client';

import { useState, useMemo, useCallback, Suspense, useTransition } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { PageHeader } from "@/components/page-header";
import {
  PlusCircle,
  FileQuestion,
  Users,
  Search,
  X,
  Database,
  Upload,
  UserPlus,
  Trash2,
  CheckCircle,
  FileText,
} from "lucide-react";
import type { AudienceList, AudienceCriterion, Prospect } from "@/lib/types";
import { useToast } from "@/hooks/use-toast";
import { industries, jobTitles } from "@/lib/mock-data";
import { Badge } from "@/components/ui/badge";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Checkbox } from "@/components/ui/checkbox";
import { cn } from "@/lib/utils";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useDropzone } from "react-dropzone";
import Papa from 'papaparse'; // Import PapaParse for client-side header parsing

// --- SERVER ACTIONS ---
import { addAudienceList, createAudienceFromCSV } from "../actions";

// --- NEW COMPONENT IMPORT ---
import { ColumnMappingDialog } from "./_components/column-mapping-dialog";


// The sub-components (AudienceBuilder, ProspectManager, etc.) do not need to change.
// You can keep your existing versions of these.
const AudienceBuilder = ({ criteria, setCriteria, disabled }: { criteria: AudienceCriterion[], setCriteria: (criteria: AudienceCriterion[]) => void, disabled?: boolean }) => {
    const [open, setOpen] = useState(false);
    const [searchTerm, setSearchTerm] = useState("");
    const [activeTab, setActiveTab] = useState<'Industry' | 'Job Title'>('Industry');
    const dataMap = { 'Industry': industries, 'Job Title': jobTitles };
    const handleToggleCriterion = (type: AudienceCriterion['type'], value: string) => {
        const newCriterion = { id: `${type}-${value}`, type, value };
        const isSelected = criteria.some(c => c.id === newCriterion.id);
        if (isSelected) {
            setCriteria(criteria.filter(c => c.id !== newCriterion.id));
        } else {
            setCriteria([...criteria, newCriterion]);
        }
    };
    const handleRemoveCriterion = (id: string) => setCriteria(criteria.filter(c => c.id !== id));
    const filteredData = useMemo(() => dataMap[activeTab].filter(item => item.toLowerCase().includes(searchTerm.toLowerCase())), [searchTerm, activeTab]);
    return (
        <div className="grid gap-4">
            <Popover open={open} onOpenChange={setOpen}>
                <PopoverTrigger asChild>
                    <Button type="button" variant="outline" className="w-full justify-start text-muted-foreground" disabled={disabled}><PlusCircle className="mr-2 h-4 w-4" /> Add audience criteria...</Button>
                </PopoverTrigger>
                <PopoverContent className="w-[400px] p-0" align="start">
                    <div className="p-4"><div className="relative"><Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" /><Input placeholder="Search criteria..." className="pl-10" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} /></div></div>
                    <Tabs defaultValue="Industry" onValueChange={(value) => setActiveTab(value as 'Industry' | 'Job Title')} className="w-full">
                        <TabsList className="grid w-full grid-cols-2 rounded-none"><TabsTrigger value="Industry">Industry</TabsTrigger><TabsTrigger value="Job Title">Job Title</TabsTrigger></TabsList>
                        <TabsContent value={activeTab}>
                            <ScrollArea className="h-64"><div className="p-4 space-y-2">
                                {filteredData.map(item => {
                                    const criterionId = `${activeTab}-${item}`;
                                    const isSelected = criteria.some(c => c.id === criterionId);
                                    return (<div key={item} className="flex items-center gap-2 cursor-pointer" onClick={() => handleToggleCriterion(activeTab, item)}><Checkbox id={criterionId} checked={isSelected} /><Label htmlFor={criterionId} className="font-normal cursor-pointer flex-1">{item}</Label></div>)
                                })}
                            </div></ScrollArea>
                        </TabsContent>
                    </Tabs>
                </PopoverContent>
            </Popover>
            <div className="flex flex-wrap items-center gap-2 min-h-[2.5rem] rounded-md border p-2 bg-muted/50"><span className="text-sm font-medium">Current Criteria:</span>{criteria.length === 0 && <span className="text-sm text-muted-foreground">None selected</span>}{criteria.map(c => (<Badge key={c.id} variant="secondary">{`${c.type}: ${c.value}`}{!disabled && (<button type="button" onClick={() => handleRemoveCriterion(c.id)} className="ml-1.5 rounded-full outline-none ring-offset-background focus:ring-2 focus:ring-ring focus:ring-offset-2"><X className="h-3 w-3 text-muted-foreground hover:text-foreground" /></button>)}</Badge>))}</div>
        </div>
    )
};
function ProspectManager({ prospects, setProspects, disabled }: { prospects: Prospect[], setProspects: (prospects: Prospect[]) => void, disabled?: boolean }) {
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const addProspect = () => { if (name && email) { setProspects([...prospects, { id: `PROS${Date.now()}`, name, email }]); setName(''); setEmail(''); } };
    const removeProspect = (id: string) => setProspects(prospects.filter(p => p.id !== id));
    return (
        <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><Users className="h-6 w-6" /></div><div><CardTitle>Prospects</CardTitle><CardDescription>Manually add prospects. Current count: {prospects.length}</CardDescription></div></div></CardHeader>
            <CardContent><div className="grid gap-4">{!disabled && (<div className="flex gap-2"><Input placeholder="Prospect Name" value={name} onChange={e => setName(e.target.value)} /><Input placeholder="Prospect Email" type="email" value={email} onChange={e => setEmail(e.target.value)} /><Button type="button" onClick={addProspect}><UserPlus className="mr-2 h-4 w-4" /> Add</Button></div>)}<Table><TableHeader><TableRow><TableHead>Name</TableHead><TableHead>Email</TableHead>{!disabled && <TableHead className="text-right">Actions</TableHead>}</TableRow></TableHeader><TableBody>{prospects.map((prospect) => (<TableRow key={prospect.id}><TableCell>{prospect.name}</TableCell><TableCell>{prospect.email}</TableCell>{!disabled && (<TableCell className="text-right"><Button type="button" variant="ghost" size="icon" onClick={() => removeProspect(prospect.id)}><Trash2 className="h-4 w-4" /></Button></TableCell>)}</TableRow>))}{prospects.length === 0 && (<TableRow><TableCell colSpan={3} className="text-center text-muted-foreground">No prospects added.</TableCell></TableRow>)}</TableBody></Table></div></CardContent>
        </Card>
    )
}
function ImportListCard({ onFileUpload, uploadedFile, disabled }: { onFileUpload: (file: File | null) => void, uploadedFile: File | null, disabled?: boolean }) {
    const { toast } = useToast();
    const onDrop = useCallback((acceptedFiles: File[]) => { if (acceptedFiles.length > 0) { onFileUpload(acceptedFiles[0]); toast({ title: 'File Uploaded', description: `${acceptedFiles[0].name} is ready.` }); } }, [onFileUpload, toast]);
    const { getRootProps, getInputProps, isDragActive } = useDropzone({ onDrop, accept: { 'text/csv': ['.csv'] }, maxFiles: 1, disabled });
    return (
        <Card>
            <CardHeader><CardTitle>Import Audience</CardTitle><CardDescription>Drag and drop a CSV file or click to upload.</CardDescription></CardHeader>
            <CardContent>
                 {uploadedFile ? (<div className="flex items-center justify-center w-full h-64 border-2 border-dashed rounded-lg bg-muted"><div className="text-center"><FileText className="w-12 h-12 mx-auto mb-4 text-green-600" /><p className="font-semibold">{uploadedFile.name}</p><p className="text-sm text-muted-foreground">{Math.round(uploadedFile.size / 1024)} KB</p><Button type="button" variant="link" size="sm" onClick={() => onFileUpload(null)}>Remove file</Button></div></div>) : (<div {...getRootProps()} className={cn("flex items-center justify-center w-full h-64 border-2 border-dashed rounded-lg cursor-pointer bg-muted hover:bg-muted/80", isDragActive && "border-primary", disabled && "cursor-not-allowed opacity-50")}><input {...getInputProps()} /><div className="flex flex-col items-center justify-center pt-5 pb-6"><Upload className="w-8 h-8 mb-4 text-muted-foreground" /><p className="mb-2 text-sm text-muted-foreground"><span className="font-semibold">Click to upload</span> or drag and drop</p><p className="text-xs text-muted-foreground">CSV only (MAX. 50MB)</p></div></div>)}
            </CardContent>
        </Card>
    );
}
function EpDatabaseCard({ onConnect, connected, disabled }: { onConnect: () => void, connected: boolean, disabled?: boolean }) {
     return (
        <Card>
            <CardHeader><CardTitle>EP Database Integration</CardTitle><CardDescription>Connect to the EP database to enrich your audience.</CardDescription></CardHeader>
            <CardContent className="flex items-center gap-4"><Database className="h-6 w-6 text-muted-foreground" /><div className="flex-1"><p className="font-semibold">Enable EP Database</p><p className="text-sm text-muted-foreground">Dynamically find prospects using BigQuery data.</p></div><Button type="button" variant={connected ? "secondary" : "outline"} onClick={onConnect} disabled={disabled || connected}>{connected && <CheckCircle className="mr-2 h-4 w-4 text-green-600" />} {connected ? "Connected" : "Connect"}</Button></CardContent>
        </Card>
    )
}


function AudiencePageContent() {
    const router = useRouter();
    const searchParams = useSearchParams();
    const { toast } = useToast();
    const [isPending, startTransition] = useTransition();

    const redirectUrl = searchParams.get('redirect');
    const [name, setName] = useState('');
    const [creationMethod, setCreationMethod] = useState<'manual' | 'import' | 'ep_database'>('manual');
    const [prospects, setProspects] = useState<Prospect[]>([]);
    const [criteria, setCriteria] = useState<AudienceCriterion[]>([]);

    // --- NEW STATE FOR CSV IMPORT FLOW ---
    const [uploadedFile, setUploadedFile] = useState<File | null>(null);
    const [csvHeaders, setCsvHeaders] = useState<string[]>([]);
    const [columnMapping, setColumnMapping] = useState<{ name: string | string[]; email: string } | null>(null);
    const [isMappingDialogOpen, setIsMappingDialogOpen] = useState(false);

    const [epConnected, setEpConnected] = useState(false);

    // --- NEW: Handle file upload and trigger mapping dialog ---
    const handleFileUpload = (file: File | null) => {
        setUploadedFile(file);
        if (file) {
            // Parse headers on the client to show the mapping dialog immediately
            Papa.parse(file, {
                preview: 1, // Only read the first row
                header: true,
                skipEmptyLines: true,
                complete: (results) => {
                    const headers = results.meta.fields || [];
                    if (headers.length > 0) {
                        setCsvHeaders(headers);
                        setIsMappingDialogOpen(true);
                    } else {
                        toast({ variant: 'destructive', title: 'Invalid CSV', description: 'Could not read headers from the file.' });
                        setUploadedFile(null);
                    }
                },
            });
        } else {
            // Reset if file is removed
            setCsvHeaders([]);
            setColumnMapping(null);
        }
    };

    const handleMappingConfirm = (mapping: { name: string | string[]; email: string }) => {
        setColumnMapping(mapping);
        toast({ title: "Mapping Confirmed", description: "Your CSV columns are now mapped." });
    };

    const handleCreateAudience = (status: 'Draft' | 'Saved') => {
      if (!name.trim()) {
        toast({ variant: 'destructive', title: 'Missing Audience Name' });
        return;
      }

      startTransition(async () => {
        // --- LOGIC TO CALL THE CORRECT SERVER ACTION ---
        if (creationMethod === 'import') {
            if (!uploadedFile || !columnMapping) {
                toast({ variant: 'destructive', title: 'Import Not Ready', description: 'Please upload a file and map the columns before saving.' });
                return;
            }
            const formData = new FormData();
            formData.append('name', name);
            formData.append('status', status);
            formData.append('criteria', JSON.stringify(criteria));
            formData.append('file', uploadedFile);
            formData.append('columnMapping', JSON.stringify(columnMapping));

            const result = await createAudienceFromCSV(null, formData);
            if (result.success) {
                toast({ title: "Audience Imported!", description: result.message });
                router.push(redirectUrl || '/audience');
            } else {
                toast({ variant: 'destructive', title: 'Import Failed', description: result.message });
            }

        } else {
            // --- HANDLE MANUAL or EP DATABASE CREATION ---
            const newListData: Omit<AudienceList, 'id'> = {
                name, status,
                source: creationMethod === 'ep_database' ? 'EP DB' : 'Manual',
                criteria,
                count: creationMethod === 'ep_database' ? 4200 : prospects.length, // Mock EP DB count
                prospects: creationMethod === 'manual' ? prospects : [],
                createdAt: new Date().toISOString(),
                lastModified: new Date().toISOString(),
            };

            const result = await addAudienceList(newListData);
            if (result.success) {
                toast({ title: "Audience List Saved!", description: result.message });
                router.push(redirectUrl || '/audience');
            } else {
                toast({ variant: 'destructive', title: 'Save Failed', description: result.message });
            }
        }
      });
    };

    const handleEpConnect = () => {
      setCreationMethod('ep_database');
      setEpConnected(true);
      toast({ title: 'EP Database Connected' });
    }

    const isCreationDisabled = epConnected || !!uploadedFile;

    return (
      <>
        <PageHeader title="Create New Audience List" description="Define a new audience to target in your campaigns." />
        <div className="space-y-8">
          <Card>
            <CardHeader><div className="flex items-center gap-4"><div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary"><FileQuestion className="h-6 w-6" /></div><div><CardTitle>Audience Details</CardTitle><CardDescription>Give your new audience list a name.</CardDescription></div></div></CardHeader>
            <CardContent><div className="grid gap-2 max-w-lg"><Label htmlFor="audience-name">Audience Name</Label><Input id="audience-name" placeholder="e.g., Q4 Enterprise Leads" value={name} onChange={(e) => setName(e.target.value)} /></div></CardContent>
          </Card>

          <Tabs value={creationMethod} onValueChange={(value) => setCreationMethod(value as any)} className="w-full">
              <TabsList className="grid w-full grid-cols-3">
                  <TabsTrigger value="manual" disabled={isCreationDisabled && creationMethod !== 'manual'}>Create Manually</TabsTrigger>
                  <TabsTrigger value="import" disabled={isCreationDisabled && creationMethod !== 'import'}>Import List</TabsTrigger>
                  <TabsTrigger value="ep_database" disabled={isCreationDisabled && creationMethod !== 'ep_database'}>EP Database</TabsTrigger>
              </TabsList>
              <TabsContent value="manual" className="space-y-8 mt-4"><ProspectManager prospects={prospects} setProspects={setProspects} /><Card><CardHeader><CardTitle>Criteria (Optional)</CardTitle><CardDescription>Further refine your manually added list.</CardDescription></CardHeader><CardContent><AudienceBuilder criteria={criteria} setCriteria={setCriteria} /></CardContent></Card></TabsContent>
              <TabsContent value="import" className="space-y-8 mt-4">
                  {/* --- USE THE NEW FILE HANDLER --- */}
                  <ImportListCard onFileUpload={handleFileUpload} uploadedFile={uploadedFile} disabled={isCreationDisabled && creationMethod !== 'import'}/>
                  <Card>
                      <CardHeader><CardTitle>Criteria (Optional)</CardTitle><CardDescription>Apply criteria to your imported list.</CardDescription></CardHeader>
                      <CardContent><AudienceBuilder criteria={criteria} setCriteria={setCriteria} disabled={isCreationDisabled && creationMethod !== 'import'} /></CardContent>
                  </Card>
              </TabsContent>
              <TabsContent value="ep_database" className="space-y-8 mt-4"><EpDatabaseCard onConnect={handleEpConnect} connected={epConnected} disabled={isCreationDisabled && creationMethod !== 'ep_database'} /><Card><CardHeader><CardTitle>Criteria</CardTitle><CardDescription>Select criteria to build your audience from the EP database.</CardDescription></CardHeader><CardContent><AudienceBuilder criteria={criteria} setCriteria={setCriteria} disabled={!epConnected} /></CardContent></Card></TabsContent>
          </Tabs>

          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => handleCreateAudience('Draft')} disabled={isPending}>
              {isPending ? "Saving..." : "Save as Draft"}
            </Button>
            <Button onClick={() => handleCreateAudience('Saved')} disabled={isPending}>
              {isPending ? "Saving..." : "Save Audience"}
            </Button>
          </div>
        </div>

        {/* --- RENDER THE MAPPING DIALOG --- */}
        <ColumnMappingDialog
          open={isMappingDialogOpen}
          onOpenChange={setIsMappingDialogOpen}
          csvHeaders={csvHeaders}
          onConfirm={handleMappingConfirm}
        />
      </>
    );
}

export default function NewAudiencePage() {
    return (
        <Suspense fallback={<div>Loading...</div>}>
            <AudiencePageContent />
        </Suspense>
    );
}

