'use client';

import { useState } from 'react';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { ArrowRight, Mails, User } from 'lucide-react';

interface ColumnMappingDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  csvHeaders: string[];
  onConfirm: (mapping: { name: string | string[]; email: string }) => void;
}

export function ColumnMappingDialog({
  open,
  onOpenChange,
  csvHeaders,
  onConfirm,
}: ColumnMappingDialogProps) {
  const [nameType, setNameType] = useState<'single' | 'double'>('single');
  const [emailMapping, setEmailMapping] = useState<string>('');
  const [fullNameMapping, setFullNameMapping] = useState<string>('');
  const [firstNameMapping, setFirstNameMapping] = useState<string>('');
  const [lastNameMapping, setLastNameMapping] = useState<string>('');

  const handleConfirm = () => {
    const name = nameType === 'single' ? fullNameMapping : [firstNameMapping, lastNameMapping];

    // Basic validation to ensure something is selected
    if (!emailMapping || (nameType === 'single' && !fullNameMapping) || (nameType === 'double' && (!firstNameMapping || !lastNameMapping))) {
        // You could add a toast notification here for better UX
        alert("Please map all required fields.");
        return;
    }

    onConfirm({ name, email: emailMapping });
    onOpenChange(false); // Close the dialog on confirmation
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[525px]">
        <DialogHeader>
          <DialogTitle>Map Your CSV Columns</DialogTitle>
          <DialogDescription>
            Match the columns from your uploaded file to the required fields in our system.
          </DialogDescription>
        </DialogHeader>
        <div className="grid gap-6 py-4">
          {/* Email Mapping */}
          <div className="flex items-center gap-4">
            <div className="flex-shrink-0 flex items-center gap-2 w-32">
              <Mails className="h-4 w-4 text-primary" />
              <Label htmlFor="email" className="font-semibold">
                Email Address*
              </Label>
            </div>
            <ArrowRight className="h-4 w-4 text-muted-foreground" />
            <Select value={emailMapping} onValueChange={setEmailMapping}>
              <SelectTrigger><SelectValue placeholder="Select CSV column..." /></SelectTrigger>
              <SelectContent>
                {csvHeaders.map((header) => (
                  <SelectItem key={header} value={header}>{header}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Name Mapping */}
          <div className="flex items-start gap-4">
            <div className="flex-shrink-0 flex items-center gap-2 w-32 pt-2">
              <User className="h-4 w-4 text-primary" />
              <Label className="font-semibold">Full Name*</Label>
            </div>
            <ArrowRight className="h-4 w-4 text-muted-foreground pt-2" />
            <div className="flex-grow space-y-4">
              <div className='flex items-center space-x-2'>
                <Button variant={nameType === 'single' ? 'secondary' : 'outline'} size="sm" onClick={() => setNameType('single')}>One Column</Button>
                <Button variant={nameType === 'double' ? 'secondary' : 'outline'} size="sm" onClick={() => setNameType('double')}>Two Columns</Button>
              </div>
              <p className="text-xs text-muted-foreground -mt-2">
                Use "Two Columns" if your CSV has separate first and last names.
              </p>

              {nameType === 'single' ? (
                <Select value={fullNameMapping} onValueChange={setFullNameMapping}>
                  <SelectTrigger><SelectValue placeholder="Select full name column..." /></SelectTrigger>
                  <SelectContent>{csvHeaders.map((h) => <SelectItem key={h} value={h}>{h}</SelectItem>)}</SelectContent>
                </Select>
              ) : (
                <div className="grid grid-cols-2 gap-2">
                   <Select value={firstNameMapping} onValueChange={setFirstNameMapping}>
                    <SelectTrigger><SelectValue placeholder="First name column..." /></SelectTrigger>
                    <SelectContent>{csvHeaders.map((h) => <SelectItem key={h} value={h}>{h}</SelectItem>)}</SelectContent>
                  </Select>
                   <Select value={lastNameMapping} onValueChange={setLastNameMapping}>
                    <SelectTrigger><SelectValue placeholder="Last name column..." /></SelectTrigger>
                    <SelectContent>{csvHeaders.map((h) => <SelectItem key={h} value={h}>{h}</SelectItem>)}</SelectContent>
                  </Select>
                </div>
              )}
            </div>
          </div>
        </div>
        <DialogFooter>
          <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
          <Button type="button" onClick={handleConfirm}>Confirm Mapping</Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

