'use server';

import { pool } from '@/lib/db';
import { revalidatePath } from 'next/cache';
import { z } from 'zod';
import Papa from 'papaparse'; // Library to parse CSV files
import { getUserId } from '@/lib/auth'; // You need a helper to get the current user's ID
import type { AudienceList } from '@/lib/types';

/**
 * Fetches all audience lists for the current user, including their prospects.
 */
export async function getAudienceLists(): Promise<AudienceList[]> {
    const userId = await getUserId();
    if (!userId) return [];
    try {
        const query = `
            SELECT
                a.*,
                (
                    SELECT json_agg(p.*)
                    FROM prospects p
                    WHERE p.audience_id = a.id
                ) as prospects
            FROM audiences a
            WHERE a.user_id = $1
            ORDER BY a.last_modified DESC
        `;
        const { rows } = await pool.query(query, [userId]);
        return rows.map(row => ({
            ...row,
            prospects: row.prospects || [], // Ensure prospects is always an array
            lastModified: row.last_modified,
            createdAt: row.created_at,
        })) as AudienceList[];
    } catch (error) {
        console.error('Database Error: Failed to fetch audience lists:', error);
        return [];
    }
}

/**
 * Adds a new audience list with manually added prospects.
 */
export async function addAudienceList(audienceData: Omit<AudienceList, 'id'>) {
    const userId = await getUserId();
    if (!userId) return { success: false, message: 'Authentication required.' };

    const { name, status, source, criteria, prospects } = audienceData;
    const client = await pool.connect();
    try {
        await client.query('BEGIN');
        const audienceInsertQuery = `
            INSERT INTO audiences (user_id, name, status, source, criteria, count, created_at, last_modified)
            VALUES ($1, $2, $3, $4, $5, $6, NOW(), NOW())
            RETURNING id;
        `;
        const audienceResult = await client.query(audienceInsertQuery, [
            userId, name, status, source, JSON.stringify(criteria), prospects?.length || 0
        ]);
        const newAudienceId = audienceResult.rows[0].id;

        if (prospects && prospects.length > 0) {
            for (const prospect of prospects) {
                await client.query(
                    'INSERT INTO prospects (audience_id, name, email) VALUES ($1, $2, $3)',
                    [newAudienceId, prospect.name, prospect.email]
                );
            }
        }
        await client.query('COMMIT');
        revalidatePath('/audience');
        return { success: true, message: 'Audience saved successfully!' };
    } catch (error) {
        await client.query('ROLLBACK');
        console.error('Database Error: Failed to add audience list:', error);
        return { success: false, message: 'Failed to save audience to database.' };
    } finally {
        client.release();
    }
}

/**
 * Creates a new audience by intelligently importing prospects from a CSV file.
 */
export async function createAudienceFromCSV(prevState: any, formData: FormData) {
  const userId = await getUserId();
  if (!userId) return { success: false, message: 'Authentication required.' };

  const schema = z.object({
    name: z.string().min(1, 'Audience name is required.'),
    status: z.enum(['Draft', 'Saved']),
    criteria: z.string(),
    file: z.instanceof(File).refine(file => file.size > 0, 'CSV file is required.'),
    columnMapping: z.string(),
  });
  const validated = schema.safeParse(Object.fromEntries(formData.entries()));

  if (!validated.success) {
    return { success: false, message: 'Invalid form data. Please check your entries.' };
  }
  const { name, status, file } = validated.data;
  const criteria = JSON.parse(validated.data.criteria);
  const mapping = JSON.parse(validated.data.columnMapping) as { name: string | string[]; email: string };

  const client = await pool.connect();
  try {
    const fileContent = await file.text();
    const parseResult = Papa.parse(fileContent, { header: true, skipEmptyLines: true });
    const prospectsData = parseResult.data as Record<string, string>[];
    if (prospectsData.length === 0) return { success: false, message: 'CSV file is empty or invalid.' };

    await client.query('BEGIN');
    const audienceRes = await client.query(`
        INSERT INTO audiences (user_id, name, status, source, criteria, count, created_at, last_modified)
        VALUES ($1, $2, $3, 'Import', $4, $5, NOW(), NOW()) RETURNING id;
    `, [userId, name, status, JSON.stringify(criteria), prospectsData.length]);
    const newAudienceId = audienceRes.rows[0].id;

    for (const row of prospectsData) {
      const email = row[mapping.email]?.trim();
      let prospectName = '';
      if (Array.isArray(mapping.name)) {
        prospectName = `${row[mapping.name[0]]?.trim() || ''} ${row[mapping.name[1]]?.trim() || ''}`.trim();
      } else {
        prospectName = row[mapping.name]?.trim();
      }
      if (email && prospectName) {
        await client.query('INSERT INTO prospects (audience_id, name, email) VALUES ($1, $2, $3) ON CONFLICT DO NOTHING', [newAudienceId, prospectName, email]);
      }
    }
    await client.query('COMMIT');
    revalidatePath('/audience');
    return { success: true, message: `Successfully imported ${prospectsData.length} prospects!` };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error("Database Error: Failed to import CSV.", error);
    return { success: false, message: 'A database error occurred during import.' };
  } finally {
    client.release();
  }
}

/**
 * --- NEW FUNCTION ---
 * Adds a single prospect to an existing audience list.
 */
export async function addProspectToDb(audienceId: number, name: string, email: string) {
    const userId = await getUserId();
    if (!userId) return { success: false, message: 'Authentication required.' };
    const client = await pool.connect();
    try {
        await client.query('BEGIN');
        // Verify user owns the audience list before inserting
        const authCheck = await client.query('SELECT id FROM audiences WHERE id = $1 AND user_id = $2', [audienceId, userId]);
        if (authCheck.rows.length === 0) {
            throw new Error('User does not have permission to modify this audience.');
        }
        // Insert the new prospect
        await client.query(
            'INSERT INTO prospects (audience_id, name, email) VALUES ($1, $2, $3) ON CONFLICT DO NOTHING',
            [audienceId, name, email]
        );
        // Update the count on the parent audience
        await client.query('UPDATE audiences SET count = (SELECT COUNT(*) FROM prospects WHERE audience_id = $1) WHERE id = $1', [audienceId]);
        await client.query('COMMIT');
        revalidatePath(`/audience/edit/${audienceId}`);
        return { success: true, message: 'Prospect added.' };
    } catch (error) {
        await client.query('ROLLBACK');
        console.error('Database Error:', error);
        return { success: false, message: 'Failed to add prospect.' };
    } finally {
        client.release();
    }
}

/**
 * --- NEW FUNCTION ---
 * Deletes a single prospect from an audience list.
 */
export async function deleteProspectFromDb(prospectId: number) {
    const userId = await getUserId();
    if (!userId) return { success: false, message: 'Authentication required.' };
    const client = await pool.connect();
    try {
        await client.query('BEGIN');
        // Get audience_id for count update and security check
        const prospectRes = await client.query('SELECT audience_id FROM prospects WHERE id = $1', [prospectId]);
        if (prospectRes.rows.length === 0) throw new Error('Prospect not found.');
        const { audience_id } = prospectRes.rows[0];

        // Verify user owns the audience list before deleting
        const authCheck = await client.query('SELECT id FROM audiences WHERE id = $1 AND user_id = $2', [audience_id, userId]);
        if (authCheck.rows.length === 0) {
            throw new Error('User does not have permission to delete from this audience.');
        }

        // Delete the prospect
        await client.query('DELETE FROM prospects WHERE id = $1', [prospectId]);
        // Update the count on the parent audience
        await client.query('UPDATE audiences SET count = (SELECT COUNT(*) FROM prospects WHERE audience_id = $1) WHERE id = $1', [audience_id]);

        await client.query('COMMIT');
        revalidatePath(`/audience/edit/${audience_id}`);
        return { success: true, message: 'Prospect deleted.' };
    } catch (error) {
        await client.query('ROLLBACK');
        console.error('Database Error:', error);
        return { success: false, message: 'Failed to delete prospect.' };
    } finally {
        client.release();
    }
}

