'use client';

import { Bar, BarChart, CartesianGrid, Line, LineChart, ResponsiveContainer, Tooltip, XAxis, YAxis, Legend } from 'recharts';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { ChartContainer, ChartTooltipContent } from '@/components/ui/chart';
import { Button } from '@/components/ui/button';
import { FileDown, Tags, Mail, Send, Filter } from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuCheckboxItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { mockOneOffAnalytics } from '@/lib/mock-data'; // Keep for one-off until that's built

// This component receives the data fetched on the server as props
export function AnalyticsCharts({ data }: { data: any }) {
  return (
    <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
      {/* Main Campaign Performance Chart - Using live data */}
      <Card className="col-span-1 lg:col-span-4">
        <CardHeader>
          <CardTitle>Overall Campaign Performance</CardTitle>
          <CardDescription>Sent, opens, and clicks over the last 7 days.</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={{}} className="h-[300px] w-full">
            <ResponsiveContainer width="100%" height="100%">
              <LineChart data={data.campaignPerformance}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="date" />
                <YAxis />
                <Tooltip content={<ChartTooltipContent />} />
                <Legend />
                <Line type="monotone" dataKey="sent" stroke="hsl(var(--secondary-foreground))" name="Sent" />
                <Line type="monotone" dataKey="opens" stroke="hsl(var(--primary))" name="Opens" />
                <Line type="monotone" dataKey="clicks" stroke="hsl(var(--accent))" name="Clicks" />
              </LineChart>
            </ResponsiveContainer>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Engagement Metrics - Using live data */}
      <Card>
          <CardHeader><CardTitle>Total Replies</CardTitle><CardDescription>From all campaigns.</CardDescription></CardHeader>
          <CardContent><p className="text-4xl font-bold">{data.totals.totalReplies.toLocaleString()}</p></CardContent>
      </Card>
      <Card>
          <CardHeader><CardTitle>Total Bounces</CardTitle><CardDescription>Hard and soft bounces.</CardDescription></CardHeader>
          <CardContent><p className="text-4xl font-bold">{data.totals.totalBounces.toLocaleString()}</p></CardContent>
      </Card>
      <Card>
          <CardHeader><CardTitle>Total Unsubscribes</CardTitle><CardDescription>Across all campaigns.</CardDescription></CardHeader>
          <CardContent><p className="text-4xl font-bold">{data.totals.totalUnsubscribes.toLocaleString()}</p></CardContent>
      </Card>
      <Card>
          <CardHeader><CardTitle>Conversion Tracking</CardTitle><CardDescription>Requires tag setup.</CardDescription></CardHeader>
          <CardContent className="flex flex-col items-center justify-center h-full">
              <Tags className="h-12 w-12 text-muted-foreground mb-2"/>
              <p className="text-sm font-semibold mb-1">No Tags Found</p>
              <Button variant="outline" size="sm">Setup Tags</Button>
          </CardContent>
      </Card>

      {/* Performance by Campaign Table - Using live data */}
      <Card className="col-span-1 lg:col-span-4">
          <CardHeader><CardTitle className='flex items-center gap-2'><Mail /> Performance by Campaign</CardTitle><CardDescription>Detailed statistics for each campaign.</CardDescription></CardHeader>
          <CardContent>
              <Table>
                  <TableHeader><TableRow><TableHead>Campaign Name</TableHead><TableHead>Status</TableHead><TableHead>Sent</TableHead><TableHead>Open Rate (%)</TableHead><TableHead>Click Rate (%)</TableHead></TableRow></TableHeader>
                  <TableBody>
                      {data.byCampaign.map((campaign: any) => (
                          <TableRow key={campaign.id}>
                              <TableCell className="font-medium">{campaign.name}</TableCell>
                              <TableCell><Badge variant={campaign.status === 'Sent' ? 'default' : 'secondary'}>{campaign.status}</Badge></TableCell>
                              <TableCell>{Number(campaign.sent).toLocaleString()}</TableCell>
                              <TableCell>{campaign.openRate}</TableCell>
                              <TableCell>{campaign.clickRate}</TableCell>
                          </TableRow>
                      ))}
                  </TableBody>
              </Table>
          </CardContent>
      </Card>

      {/* One-off Email Performance - Still using mock data */}
      <Card className="col-span-1 lg:col-span-4">
          <CardHeader><CardTitle className='flex items-center gap-2'><Send /> One-Off Email Performance</CardTitle><CardDescription>Statistics for individual email sends.</CardDescription></CardHeader>
          <CardContent>
              <Table>
                  <TableHeader><TableRow><TableHead>Recipient</TableHead><TableHead>Subject</TableHead><TableHead>Sent Date</TableHead><TableHead>Opened</TableHead><TableHead>Clicked</TableHead></TableRow></TableHeader>
                  <TableBody>
                      {mockOneOffAnalytics.map((email) => (
                          <TableRow key={email.id}>
                              <TableCell className="font-medium">{email.recipient}</TableCell>
                              <TableCell>{email.subject}</TableCell>
                              <TableCell>{email.sentDate}</TableCell>
                              <TableCell>{email.opened ? 'Yes' : 'No'}</TableCell>
                              <TableCell>{email.clicked ? 'Yes' : 'No'}</TableCell>
                          </TableRow>
                      ))}
                  </TableBody>
              </Table>
          </CardContent>
      </Card>
    </div>
  );
}
