
'use client';

import { PageHeader } from "@/components/page-header";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { MoreHorizontal, Shield, User, BarChart, Settings, Mail, Code, DollarSign, Search, PlusCircle, Trash2, X, Activity } from "lucide-react";
import { useUsers } from "@/context/UserContext";
import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from "@/components/ui/alert-dialog";
import { useToast } from "@/hooks/use-toast";
import type { User as UserType, UserRole, UserStatus } from "@/lib/types";
import Link from "next/link";
import { cn } from "@/lib/utils";
import { useActivityLog } from "@/context/ActivityLogContext";
import { signup } from "@/app/signup/actions";

function CreateUserDialog() {
    const { toast } = useToast();
    const { fetchUsers } = useUsers();
    const { addLog, fetchLogs } = useActivityLog();
    const [open, setOpen] = useState(false);
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [role, setRole] = useState<UserRole>('Consumer');

    const handleCreateUser = async () => {
        if (!name || !email || !password) {
            toast({ variant: 'destructive', title: 'Missing fields', description: 'Please fill all fields.' });
            return;
        }

        const formData = new FormData();
        formData.append('fullName', name);
        formData.append('email', email);
        formData.append('password', password);
        formData.append('role', role);

        const result = await signup(formData);

        if (result?.error) {
            toast({ variant: 'destructive', title: 'User Creation Failed', description: result.error });
        } else if (result?.user) {
            await fetchUsers(); // Refresh the user list
            
            const adminUser = sessionStorage.getItem('userName') || 'Admin';
            
            await addLog({
                action: 'User Created',
                userId: result.user.id,
                userName: adminUser, 
                details: `New user account created for ${email} with role ${role} by ${adminUser}.`
            });

            await fetchLogs();

            toast({ title: 'User Created', description: `${name} has been added.` });
            setOpen(false);
            setName('');
            setEmail('');
            setPassword('');
            setRole('Consumer');
        }
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                <Button>
                    <PlusCircle className="mr-2 h-4 w-4" />
                    Create User
                </Button>
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>Create New User</DialogTitle>
                    <DialogDescription>
                        Fill in the details to create a new user account.
                    </DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4">
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="name" className="text-right">Full Name</Label>
                        <Input id="name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="email" className="text-right">Email</Label>
                        <Input id="email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="password" className="text-right">Password</Label>
                        <Input id="password" type="password" value={password} onChange={(e) => setPassword(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="role" className="text-right">Role</Label>
                        <Select onValueChange={(value) => setRole(value as UserRole)} defaultValue={role}>
                            <SelectTrigger className="col-span-3">
                                <SelectValue placeholder="Select a role" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="Consumer">Consumer</SelectItem>
                                <SelectItem value="Staff">Staff</SelectItem>
                                <SelectItem value="Manager">Manager</SelectItem>
                            </SelectContent>
                        </Select>
                    </div>
                </div>
                <DialogFooter>
                    <Button onClick={handleCreateUser}>Create User</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function EditUserDialog({ user }: { user: UserType }) {
    const { toast } = useToast();
    const { updateUser, fetchUsers } = useUsers();
    const { addLog, fetchLogs } = useActivityLog();
    const [open, setOpen] = useState(false);
    const [name, setName] = useState(user.name);
    const [email, setEmail] = useState(user.email);
    const [role, setRole] = useState(user.role);

    const handleUpdateUser = async () => {
        const updatedUserData: Partial<UserType> = {};
        const logDetails = [];
        if (name !== user.name) {
            updatedUserData.name = name;
            logDetails.push(`Name changed to "${name}".`);
        }
        if (email !== user.email) {
            updatedUserData.email = email;
            logDetails.push(`Email changed to "${email}".`);
        }
        if (role !== user.role) {
            updatedUserData.role = role;
            logDetails.push(`Role changed to "${role}".`);
        }

        if (Object.keys(updatedUserData).length > 0) {
            await updateUser(user.id, updatedUserData);
            await fetchUsers();
            
            const adminUser = sessionStorage.getItem('userName') || 'Admin';
            await addLog({
                action: 'User Modified',
                userId: user.id,
                userName: adminUser,
                details: `Admin ${adminUser} updated details for ${user.name} (${user.email}). ${logDetails.join(' ')}`
            });
            await fetchLogs();

            toast({ title: 'User Updated', description: `${name}'s details have been updated.` });
        } else {
            toast({ title: 'No changes', description: 'No details were changed.'})
        }
        
        setOpen(false);
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                <DropdownMenuItem onSelect={(e) => e.preventDefault()}>Edit User</DropdownMenuItem>
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>Edit User</DialogTitle>
                    <DialogDescription>
                        Update details for {user.name}.
                    </DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4">
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="name" className="text-right">Full Name</Label>
                        <Input id="name" value={name} onChange={(e) => setName(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="email" className="text-right">Email</Label>
                        <Input id="email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} className="col-span-3" />
                    </div>
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="role" className="text-right">Role</Label>
                        <Select 
                            onValueChange={(value) => setRole(value as UserRole)} 
                            defaultValue={role}
                            disabled={user.role === 'Admin'}
                        >
                            <SelectTrigger className="col-span-3">
                                <SelectValue placeholder="Select a role" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="Consumer">Consumer</SelectItem>
                                <SelectItem value="Staff">Staff</SelectItem>
                                <SelectItem value="Manager">Manager</SelectItem>
                                {user.role === 'Admin' && <SelectItem value="Admin">Admin</SelectItem>}
                            </SelectContent>
                        </Select>
                    </div>
                </div>
                <DialogFooter>
                    <Button onClick={handleUpdateUser}>Save Changes</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function ManageCreditsDialog({ user }: { user: UserType }) {
    const { toast } = useToast();
    const { updateUser, fetchUsers } = useUsers();
    const { addLog, fetchLogs } = useActivityLog();
    const [open, setOpen] = useState(false);
    const [credits, setCredits] = useState(user.credits);

    const handleUpdateCredits = async () => {
        await updateUser(user.id, { credits });
        await fetchUsers();

        const adminUser = sessionStorage.getItem('userName') || 'Admin';
        await addLog({
            action: 'Credits Managed',
            userId: user.id,
            userName: adminUser,
            details: `Credits for ${user.name} changed from ${user.credits.toLocaleString()} to ${credits.toLocaleString()} by ${adminUser}.`
        });
        await fetchLogs();

        toast({ title: 'Credits Updated', description: `${user.name}'s credits are now ${credits.toLocaleString()}.` });
        setOpen(false);
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                <DropdownMenuItem onSelect={(e) => e.preventDefault()} disabled={user.role === 'Admin'}>
                    Manage Credits
                </DropdownMenuItem>
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>Manage Credits</DialogTitle>
                    <DialogDescription>
                        Adjust credits based on subscription tiers or usage for {user.name}.
                    </DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4">
                    <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="credits" className="text-right">Credits</Label>
                        <Input id="credits" type="number" value={credits} onChange={(e) => setCredits(Number(e.target.value))} className="col-span-3" />
                    </div>
                    {user.role === 'Admin' && <p className="text-sm text-muted-foreground text-center col-span-4">Admin has unlimited credits.</p>}
                </div>
                <DialogFooter>
                    <Button onClick={handleUpdateCredits}>Update Credits</Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function SuspendUserDialog({ user }: { user: UserType }) {
    const { toast } = useToast();
    const { deleteUser, fetchUsers } = useUsers();
    const { addLog, fetchLogs } = useActivityLog();

    const handleSuspend = async () => {
        const adminUser = sessionStorage.getItem('userName') || 'Admin';
        await addLog({
            action: 'User Suspended',
            userId: user.id,
            userName: adminUser, 
            details: `User account for ${user.email} has been suspended and removed by ${adminUser}.`
        });

        await deleteUser(user.id);
        await fetchUsers();
        await fetchLogs();

        toast({ title: 'User Suspended', description: `${user.name} has been removed from the system.` });
    };

    return (
        <AlertDialog>
            <AlertDialogTrigger asChild>
                <DropdownMenuItem onSelect={(e) => e.preventDefault()} className="text-destructive" disabled={user.role === 'Admin'}>
                    Suspend User
                </DropdownMenuItem>
            </AlertDialogTrigger>
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>Are you sure you want to suspend this user?</AlertDialogTitle>
                    <AlertDialogDescription>
                        This action cannot be undone. This will permanently remove {user.name} and their data from the servers.
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <AlertDialogFooter>
                    <AlertDialogCancel>Cancel</AlertDialogCancel>
                    <AlertDialogAction onClick={handleSuspend}>Suspend</AlertDialogAction>
                </AlertDialogFooter>
            </AlertDialogContent>
        </AlertDialog>
    );
}

function UserRow({ user }: { user: UserType }) {
    const status: UserStatus = 'Active';

    const roleColors: { [key in UserRole]: string } = {
        Admin: 'bg-red-500 text-white',
        Manager: 'bg-purple-500 text-white',
        Staff: 'bg-blue-500 text-white',
        Consumer: 'bg-gray-200 text-gray-800',
    };

    return (
        <TableRow>
            <TableCell className="font-medium">{user.name}</TableCell>
            <TableCell>{user.email}</TableCell>
            <TableCell><Badge className={roleColors[user.role]}>{user.role}</Badge></TableCell>
            <TableCell>{user.credits < 0 ? 'Unlimited' : user.credits.toLocaleString()}</TableCell>
            <TableCell>{user.campaignCount || 0}</TableCell>
            <TableCell>
                <Badge
                    variant={status === 'Active' ? 'default' : 'secondary'}
                    className={cn(
                        status === 'Active' && 'bg-green-600 text-white'
                    )}
                >
                    {status}
                </Badge>
            </TableCell>
            <TableCell className="text-right">
                <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                        <Button variant="ghost" className="h-8 w-8 p-0">
                            <span className="sr-only">Open menu</span>
                            <MoreHorizontal className="h-4 w-4" />
                        </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                        <EditUserDialog user={user} />
                        <ManageCreditsDialog user={user} />
                        <SuspendUserDialog user={user} />
                    </DropdownMenuContent>
                </DropdownMenu>
            </TableCell>
        </TableRow>
    );
}


export default function AdminPage() {
    const { users } = useUsers();
    const [searchTerm, setSearchTerm] = useState('');

    const filteredUsers = users.filter(user =>
        user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        user.email.toLowerCase().includes(searchTerm.toLowerCase())
    );

    return (
        <>
            <PageHeader
                title="Admin Panel"
                description="Manage users, campaigns, and system settings."
            >
                <CreateUserDialog />
            </PageHeader>

            <div className="grid gap-6">
                <Card>
                    <CardHeader>
                        <CardTitle>User Management</CardTitle>
                        <CardDescription>View, create, and manage user accounts.</CardDescription>
                        <div className="relative pt-2">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                            <Input
                                placeholder="Search users by name or email..."
                                className="pl-10 pr-10"
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                            />
                             {searchTerm && (
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    className="absolute right-2 top-1/2 -translate-y-1/2 h-6 w-6"
                                    onClick={() => setSearchTerm('')}
                                >
                                    <X className="h-4 w-4" />
                                </Button>
                            )}
                        </div>
                    </CardHeader>
                    <CardContent>
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>User</TableHead>
                                    <TableHead>Email</TableHead>
                                    <TableHead>Role</TableHead>
                                    <TableHead>Credits</TableHead>
                                    <TableHead>No. of Campaigns</TableHead>
                                    <TableHead>Status</TableHead>
                                    <TableHead className="text-right">Actions</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {filteredUsers.length > 0 ? (
                                    filteredUsers.map(user => (
                                        <UserRow key={user.id} user={user} />
                                    ))
                                ) : (
                                    <TableRow>
                                        <TableCell colSpan={7} className="h-24 text-center">
                                            No users found.
                                        </TableCell>
                                    </TableRow>
                                )}
                            </TableBody>
                        </Table>
                    </CardContent>
                </Card>

                <div className="grid md:grid-cols-2 gap-6">
                    <Card>
                        <CardHeader>
                            <CardTitle>System Controls</CardTitle>
                            <CardDescription>Global settings and configurations.</CardDescription>
                        </CardHeader>
                        <CardContent className="grid grid-cols-2 gap-4">
                            <Button variant="outline"><Shield className="mr-2" /> Domain Warm-up Status</Button>
                            <Button variant="outline"><Settings className="mr-2" /> SMTP Pool Management</Button>
                            <Button variant="outline"><Code className="mr-2" /> AI Prompts Control</Button>
                            <Button variant="outline"><DollarSign className="mr-2" /> API Tier Management</Button>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader>
                            <CardTitle>System Health & Logs</CardTitle>
                            <CardDescription>Monitor system performance and view logs.</CardDescription>
                        </CardHeader>
                        <CardContent className="grid grid-cols-2 gap-4">
                             <Link href="/admin/all-campaigns">
                                <Button variant="outline" className="w-full"><Mail className="mr-2" /> View All Campaigns</Button>
                            </Link>
                             <Link href="/admin/sending-engine-logs">
                                <Button variant="outline" className="w-full"><BarChart className="mr-2" /> Sending Engine Logs</Button>
                            </Link>
                             <Link href="/admin/api-usage">
                                <Button variant="outline" className="w-full"><User className="mr-2" /> API Usage Logs</Button>
                             </Link>
                             <Link href="/admin/activity-logs">
                                <Button variant="outline" className="w-full"><Activity className="mr-2" /> Activity Logs</Button>
                            </Link>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </>
    );
}
