
'use client';

import { useState } from 'react';
import { PageHeader } from "@/components/page-header";
import { useCampaigns } from "@/context/CampaignContext";
import { useUsers } from "@/context/UserContext";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { cn } from "@/lib/utils";
import Link from "next/link";
import { Button } from "@/components/ui/button";
import { Eye, ArrowLeft, Search } from "lucide-react";
import type { User, UserRole, Campaign } from '@/lib/types';
import { Input } from '@/components/ui/input';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { EditCampaignDialog } from '@/components/edit-campaign-dialog';

function UserCampaignsDialog({ user, campaigns, open, onOpenChange, onCampaignView }: { user: User | null; campaigns: Campaign[]; open: boolean; onOpenChange: (open: boolean) => void; onCampaignView: (campaign: Campaign) => void; }) {
    if (!user) return null;

    const statusColors: { [key: string]: string } = {
        Sent: "bg-blue-100 text-blue-800",
        Scheduled: "bg-yellow-100 text-yellow-800",
        Draft: "bg-gray-100 text-gray-800",
    };

    const campaignsForUser = campaigns.filter(c => c.userId === user.id);

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="max-w-3xl">
                <DialogHeader>
                    <DialogTitle>Campaigns for {user.name}</DialogTitle>
                    <DialogDescription>
                        A list of all campaigns created by {user.name}.
                    </DialogDescription>
                </DialogHeader>
                <div className="py-4 max-h-[60vh] overflow-y-auto">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Campaign Name</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead>Recipients</TableHead>
                                <TableHead className="text-right">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {campaignsForUser.map(campaign => (
                                <TableRow key={campaign.id}>
                                    <TableCell className="font-medium">{campaign.name}</TableCell>
                                    <TableCell>
                                        <Badge className={cn("border-transparent", statusColors[campaign.status])}>
                                            {campaign.status}
                                        </Badge>
                                    </TableCell>
                                    <TableCell>{campaign.audience.count.toLocaleString()}</TableCell>
                                    <TableCell className="text-right">
                                        <Button variant="outline" size="sm" onClick={() => onCampaignView(campaign)}>
                                            <Eye className="mr-2 h-4 w-4" />
                                            View Campaign
                                        </Button>
                                    </TableCell>
                                </TableRow>
                            ))}
                            {campaignsForUser.length === 0 && (
                                <TableRow>
                                    <TableCell colSpan={4} className="h-24 text-center text-muted-foreground">
                                        This user has not created any campaigns.
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </div>
            </DialogContent>
        </Dialog>
    );
}


export default function AllCampaignsPage() {
    const { campaigns, updateCampaign } = useCampaigns();
    const { users } = useUsers();
    const [selectedUser, setSelectedUser] = useState<User | null>(null);
    const [isUserCampaignsOpen, setIsUserCampaignsOpen] = useState(false);
    const [campaignToView, setCampaignToView] = useState<Campaign | null>(null);
    const [isCampaignDetailOpen, setIsCampaignDetailOpen] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');

    const roleColors: { [key in UserRole]: string } = {
        Admin: 'bg-red-500 text-white',
        Manager: 'bg-purple-500 text-white',
        Staff: 'bg-blue-500 text-white',
        Consumer: 'bg-gray-200 text-gray-800',
    };

    const userCampaigns = users
        .map(user => ({
            ...user,
            campaignCount: campaigns.filter(c => c.userId === user.id && c.status === 'Sent').length,
        }))
        .filter(user => user.name.toLowerCase().includes(searchTerm.toLowerCase()));
    
    const handleViewUserCampaignsClick = (user: User) => {
        setSelectedUser(user);
        setIsUserCampaignsOpen(true);
    };

    const handleViewCampaignClick = (campaign: Campaign) => {
        setCampaignToView(campaign);
        setIsCampaignDetailOpen(true);
        setIsUserCampaignsOpen(false); // Close the first dialog
    }
    
    const handleCampaignDialogClose = () => {
        setIsCampaignDetailOpen(false);
        setCampaignToView(null);
        // Re-open the user campaigns dialog if a user was selected
        if (selectedUser) {
            setIsUserCampaignsOpen(true);
        }
    }

    return (
        <div className="flex flex-col h-full">
            <PageHeader
                title="All User Campaigns"
                description="A comprehensive list of all users and their campaign activity."
            />
            <Card className="flex-1">
                <CardHeader>
                    <CardTitle>Users Overview</CardTitle>
                    <div className="relative pt-2">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                        <Input
                            placeholder="Search users..."
                            className="pl-10"
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                        />
                    </div>
                </CardHeader>
                <CardContent>
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>User Name</TableHead>
                                <TableHead>Role</TableHead>
                                <TableHead>No. of Campaigns</TableHead>
                                <TableHead className="text-right">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {userCampaigns.map(user => (
                                <TableRow key={user.id}>
                                    <TableCell className="font-medium">{user.name}</TableCell>
                                    <TableCell>
                                        <Badge className={roleColors[user.role]}>{user.role}</Badge>
                                    </TableCell>
                                    <TableCell>{user.campaignCount}</TableCell>
                                    <TableCell className="text-right">
                                        <Button variant="outline" size="sm" onClick={() => handleViewUserCampaignsClick(user)}>
                                            View Campaign List
                                        </Button>
                                    </TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
             <div className="flex justify-end mt-6">
                <Link href="/admin">
                    <Button variant="outline">
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Admin Panel
                    </Button>
                </Link>
            </div>
            <UserCampaignsDialog 
                user={selectedUser}
                campaigns={campaigns}
                open={isUserCampaignsOpen}
                onOpenChange={setIsUserCampaignsOpen}
                onCampaignView={handleViewCampaignClick}
            />
            {campaignToView && (
                <EditCampaignDialog 
                    campaign={campaignToView}
                    open={isCampaignDetailOpen}
                    onOpenChange={handleCampaignDialogClose}
                    onSave={() => {}} // No-op for admin view
                    isAdminView={true}
                />
            )}
        </div>
    );
}
