'use server';

import { pool } from '@/lib/db';
import type { User } from '@/lib/types';
import { revalidatePath } from 'next/cache';
import sql from 'sql-template-strings';

export async function getUsers(): Promise<User[]> {
  try {
    const { rows } = await pool.query(sql`
      SELECT
        u.id,
        u.name,
        u.email,
        u.role,
        u.credits,
        u.api_key,
        u.profile_pic_url,
        COUNT(c.id) AS "campaignCount"
      FROM users u
      LEFT JOIN campaigns c ON u.id = c.user_id
      GROUP BY u.id, u.name, u.email, u.role, u.credits, u.api_key, u.profile_pic_url
      ORDER BY u.name ASC
    `);
    return rows.map(row => ({
        id: row.id,
        name: row.name,
        email: row.email,
        role: row.role,
        credits: row.credits,
        apiKey: row.api_key,
        profile_pic_url: row.profile_pic_url,
        campaignCount: parseInt(row.campaignCount, 10) // Convert count to integer
    })) as User[]; // Type assertion because campaignCount is added
  } catch (error) {
    console.error('Failed to fetch users:', error);
    return [];
  }
}

export async function getCurrentUser(email: string): Promise<User | null> {
    if (!email) return null;
    try {
        const { rows } = await pool.query(sql`SELECT id, name, email, role, credits, api_key, profile_pic_url FROM users WHERE email = ${email}`);
        if (rows.length > 0) {
            const row = rows[0];
            return {
                id: row.id,
                name: row.name,
                email: row.email,
                role: row.role,
                credits: row.credits,
                apiKey: row.api_key,
                profile_pic_url: row.profile_pic_url
            };
        }
        return null;
    } catch (error) {
        console.error('Failed to fetch current user:', error);
        return null;
    }
}

export async function updateUser(userId: string, userData: Partial<User>) {
  const { name, email, role, credits, apiKey, profile_pic_url } = userData;
  try {
    await pool.query(sql`
        UPDATE users
        SET
            name = COALESCE(${name}, name),
            email = COALESCE(${email}, email),
            role = COALESCE(${role}, role),
            credits = COALESCE(${credits}, credits),
            api_key = COALESCE(${apiKey}, api_key),
            profile_pic_url = COALESCE(${profile_pic_url}, profile_pic_url)
        WHERE id = ${userId}
    `);
    revalidatePath('/admin');
    revalidatePath('/settings');
  } catch (error) {
    console.error('Failed to update user:', error);
    throw new Error('Could not find user to update.');
  }
}

export async function deleteUser(userId: string) {
   try {
    await pool.query(sql`DELETE FROM users WHERE id = ${userId}`);
    revalidatePath('/admin');
   } catch (error) {
    console.error('Failed to delete user:', error);
    throw new Error('Could not find user to delete.');
   }
}
