'use server';

/**
 * @fileOverview Email draft generation flow using Genkit.
 *
 * - generateEmailDraft - A function that generates an email draft based on a prompt.
 * - GenerateEmailDraftInput - The input type for the generateEmailDraft function.
 * - GenerateEmailDraftOutput - The return type for the generateEmailDraft function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const GenerateEmailDraftInputSchema = z.object({
  product: z.string().describe("The product or service being promoted."),
  audience: z.string().describe("The ideal customer or target audience."),
  brandName: z.string().describe("The user's brand or company name."),
  authorName: z.string().describe("The name of the email author."),
  ctaLink: z.string().url().describe("The call-to-action link."),
  logoUrl: z.string().optional().describe("A URL to the company's logo.")
});
export type GenerateEmailDraftInput = z.infer<typeof GenerateEmailDraftInputSchema>;

const GenerateEmailDraftOutputSchema = z.object({
  subject: z.string().describe('The subject line of the email.'),
  body: z.string().describe('The body of the email in markdown format.'),
  preheader: z.string().optional().describe('The preheader text of the email, if any.'),
  cta: z.string().optional().describe('The call to action text of the email, if any.'),
  imageRecommendation: z.string().optional().describe('A recommendation for an image to include in the email, if any.'),
});
export type GenerateEmailDraftOutput = z.infer<typeof GenerateEmailDraftOutputSchema>;

export async function generateEmailDraft(input: GenerateEmailDraftInput): Promise<GenerateEmailDraftOutput> {
  return generateEmailDraftFlow(input);
}

const generateEmailDraftPrompt = ai.definePrompt({
  name: 'generateEmailDraftPrompt',
  input: {schema: GenerateEmailDraftInputSchema},
  output: {schema: GenerateEmailDraftOutputSchema},
  prompt: `You are an expert cold email copywriter. Your goal is to write a compelling, concise, and effective cold email.

Here is the information provided by the user:
- Product/Service: {{{product}}}
- Target Audience: {{{audience}}}
- Brand Name: {{{brandName}}}
- Email Author: {{{authorName}}}
- Call-to-Action Link: {{{ctaLink}}}
{{#if logoUrl}}- Logo URL: {{{logoUrl}}}{{/if}}

Based on this information, generate an email draft. The email should be friendly, professional, and persuasive. 
The body should be formatted in simple markdown.
Create a compelling Call to Action (CTA) text related to the provided link.
If a logo was provided, you can mention it.
Optionally, recommend a relevant stock photo to include in the email.

Return the email draft in JSON format.`,
});

const generateEmailDraftFlow = ai.defineFlow(
  {
    name: 'generateEmailDraftFlow',
    inputSchema: GenerateEmailDraftInputSchema,
    outputSchema: GenerateEmailDraftOutputSchema,
  },
  async input => {
    const {output} = await generateEmailDraftPrompt(input);
    return output!;
  }
);
