/**
 * Copyright 2024 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Invoke and stream response from a deployed flow.
 *
 * For example:
 *
 * ```js
 * import { streamFlow } from 'genkit/beta/client';
 *
 * const response = streamFlow({
 *   url: 'https://my-flow-deployed-url',
 *   input: 'foo',
 * });
 * for await (const chunk of response.stream) {
 *   console.log(chunk);
 * }
 * console.log(await response.output);
 * ```
 */
declare function streamFlow<O = any, S = any>({ url, input, headers, }: {
    /** URL of the deployed flow. */
    url: string;
    /** Flow input. */
    input?: any;
    /** A map of HTTP headers to be added to the HTTP call. */
    headers?: Record<string, string>;
}): {
    readonly output: Promise<O>;
    readonly stream: AsyncIterable<S>;
};
/**
 * Invoke a deployed flow over HTTP(s).
 *
 * For example:
 *
 * ```js
 * import { runFlow } from 'genkit/beta/client';
 *
 * const response = await runFlow({
 *   url: 'https://my-flow-deployed-url',
 *   input: 'foo',
 * });
 * console.log(await response);
 * ```
 */
declare function runFlow<O = any>({ url, input, headers, }: {
    /** URL of the deployed flow. */
    url: string;
    /** Flow input. */
    input?: any;
    /** A map of HTTP headers to be added to the HTTP call. */
    headers?: Record<string, string>;
}): Promise<O>;

export { runFlow, streamFlow };
