import dotenv from 'dotenv';
dotenv.config({ path: '.env.local' });

import { pool } from './src/lib/db';
import nodemailer from 'nodemailer';
import cron from 'node-cron';
import { JSDOM } from 'jsdom';

console.log("🚀 Delivery Worker starting...");

const DEFAULT_TRACKING_DOMAIN = process.env.TRACKING_DOMAIN;

function injectTracking(htmlBody: string, campaignId: number, prospectEmail: string, trackingDomain: string): string {
    const dom = new JSDOM(htmlBody);
    const document = dom.window.document;
    const links = document.querySelectorAll('a');
    links.forEach(link => {
        const originalUrl = link.href;
        if (originalUrl) {
            const trackingUrl = `https://${trackingDomain}/api/track/click?cid=${campaignId}&pid=${encodeURIComponent(prospectEmail)}&url=${encodeURIComponent(originalUrl)}`;
            link.href = trackingUrl;
        }
    });
    const trackingPixel = document.createElement('img');
    const pixelUrl = `https://${trackingDomain}/api/track/open?cid=${campaignId}&pid=${encodeURIComponent(prospectEmail)}`;
    trackingPixel.src = pixelUrl;
    trackingPixel.width = 1;
    trackingPixel.height = 1;
    trackingPixel.alt = '';
    document.body.appendChild(trackingPixel);
    return dom.serialize();
}

async function processScheduledCampaigns() {
    console.log(`\n[${new Date().toISOString()}] Checking for scheduled campaigns...`);
    const client = await pool.connect();

    try {
        const campaignsRes = await client.query(`
            SELECT c.*, td.domain_name AS tracking_domain_override FROM campaigns c
            LEFT JOIN tracking_domains td ON c.tracking_domain_id = td.id
            WHERE c.status = 'Scheduled' LIMIT 10
        `);
        if (campaignsRes.rows.length === 0) {
            console.log("✅ No scheduled campaigns to process.");
            return;
        }

        for (const campaign of campaignsRes.rows) {
            console.log(`\n--- Processing Campaign: "${campaign.name}" (ID: ${campaign.id}) ---`);

            // --- The entire sending process is now wrapped in its own try/catch ---
            try {
                const trackingDomain = campaign.tracking_domain_override || DEFAULT_TRACKING_DOMAIN;
                if (!trackingDomain) throw new Error("No valid tracking domain found.");
                console.log(`   -> Using tracking domain: ${trackingDomain}`);

                if (!campaign.sequence?.steps?.length) throw new Error("Campaign has no sequence steps.");
                const firstStep = campaign.sequence.steps[0];

                await client.query("UPDATE campaigns SET status = 'Sending' WHERE id = $1", [campaign.id]);

                const prospectsRes = await client.query("SELECT email FROM prospects WHERE audience_id = $1", [campaign.audience_list_id]);
                const prospects = prospectsRes.rows;
                if (prospects.length === 0) throw new Error("Campaign has no prospects.");

                const sendersRes = await client.query(`
                    SELECT s.id, s.from_name, s.from_email, s.smtp_host, s.smtp_port, s.smtp_username, s.smtp_password
                    FROM smtp_services s JOIN campaign_senders cs ON s.id = cs.smtp_service_id
                    WHERE cs.campaign_id = $1
                `, [campaign.id]);
                const senders = sendersRes.rows;
                if (senders.length === 0) throw new Error("Campaign has no senders configured.");

                // --- NEW: Connection Verification Step ---
                console.log("   -> Verifying SMTP connections...");
                for(const sender of senders) {
                    const transporter = nodemailer.createTransport({ host: sender.smtp_host, port: sender.smtp_port, secure: sender.smtp_port === 465, auth: { user: sender.smtp_username, pass: sender.smtp_password }});
                    await transporter.verify();
                    console.log(`      ✅ Connection successful for ${sender.from_email || sender.smtp_username}`);
                }

                console.log(`✉️  Preparing to send to ${prospects.length} prospects...`);
                let successfulSends = 0;
                let senderIndex = 0;

                for (const prospect of prospects) {
                    const currentSender = senders[senderIndex];
                    senderIndex = (senderIndex + 1) % senders.length;

                    try {
                        const transporter = nodemailer.createTransport({ host: currentSender.smtp_host, port: currentSender.smtp_port, secure: currentSender.smtp_port === 465, auth: { user: currentSender.smtp_username, pass: currentSender.smtp_password }});
                        if (!firstStep.subject || !firstStep.body) throw new Error("Sequence step is missing a subject or body.");

                        const trackedHtml = injectTracking(firstStep.body, campaign.id, prospect.email, trackingDomain);
                        const fromName = currentSender.from_name || currentSender.smtp_username;
                        const fromEmail = currentSender.from_email || currentSender.smtp_username;

                        await transporter.sendMail({ from: `"${fromName}" <${fromEmail}>`, to: prospect.email, subject: firstStep.subject, html: trackedHtml });
                        await client.query( `INSERT INTO email_events (campaign_id, prospect_email, smtp_service_id, event_type) VALUES ($1, $2, $3, 'SENT')`, [campaign.id, prospect.email, currentSender.id]);
                        successfulSends++;
                    } catch (error) {
                        // --- ENHANCED ERROR LOGGING ---
                        // This will now log the complete error object for detailed debugging.
                        console.error(`   -> FAILED to send to ${prospect.email}. Full Error:`, error);
                    }
                }
                await client.query("UPDATE campaigns SET status = 'Sent' WHERE id = $1", [campaign.id]);
                console.log(`--- ✅ Campaign "${campaign.name}" complete. Sent: ${successfulSends}/${prospects.length} ---`);

            } catch (campaignError) {
                // This block catches errors in the setup (like verification) and fails the whole campaign.
                console.error(`❌ A critical error occurred while processing campaign ${campaign.id}. Marking as failed. Reason:`, campaignError instanceof Error ? campaignError.message : campaignError);
                await client.query("UPDATE campaigns SET status = 'Failed' WHERE id = $1", [campaign.id]);
            }
        }
    } catch (error) {
        console.error("❌ A fatal error occurred during the main worker process:", error);
    } finally {
        client.release();
    }
}

cron.schedule('* * * * *', processScheduledCampaigns);
console.log("🕒 Worker scheduled to run every minute.");
